#region References

using System;
using System.Collections;
using System.Data;
using System.Reflection;
using System.Text;

using gov.va.med.vbecs.DAL;
using gov.va.med.vbecs.DAL.VAL;
using gov.va.med.vbecs.ExceptionManagement;
using gov.va.med.vbecs.Common;
using TABLE = gov.va.med.vbecs.Common.VbecsTables;
using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;
#endregion

namespace gov.va.med.vbecs.BOL
{
	#region Header

	///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	///		///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	///<Developers>
	///	<Developer>Cameron Taylor</Developer>
	///</Developers>
	///<SiteName>Hines OIFO</SiteName>
	///<CreationDate>9/5/2002</CreationDate>
	///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	///<summary>
	/// Patient business object layer class.
	///</summary>

	#endregion

	public class Patient: BaseBusinessObject
	{
		#region Variables

		private const string NoABORhResults = "Information.Common.NoABORhResults";
		private const string PatienteXMDisabledNoSpecimenTesting = "Information.Common.PatienteXMDisabledNoSpecimenTesting";
		private const string PatienteXMDisabledPositiveAntibody	= "Information.Common.PatienteXMDisabledPositiveAntibody";
		private const string PatienteXMDisabledAntibodyHistory = "Information.Common.PatienteXMDisabledAntibodyHistory";
		private const string PatienteXMDisabledAntigenNegative = "Information.Common.PatienteXMDisabledAntigenNegative";
		private const string PatienteXMDisabledTypingDifficulty = "Information.Common.PatienteXMDisabledTypingDifficulty";
		private const string PatienteXMDisabledABORhDiscrepancy = "Information.Common.PatienteXMDisabledABORhDiscrepancy";
		private const string PatienteXMDisabledNeedTwoTypings = "Information.Common.PatienteXMDisabledNeedTwoTypings";
		private const string PatienteXMEnabled = "Information.Common.PatienteXMEnabled";

		private string _displayPatientID;
		private string _firstName;
		private string _lastName;
		private string _middleInitial;
		private string _patientDOBCode;
		private string _patientServiceBranch;
		private string _specimenUid;
		private string _ssn;

		private Guid _patientGUID;
		private Guid _finalMergeToPatient;
		private Guid _patientMergeGroup;

		private DateTime _dob;
		private DateTime _patientDeathDate;

		private ArrayList _aboHistory;

		private Common.ABORhSource _aboRhSource = Common.ABORhSource.None;
		private Common.Sex _sex;

		private BOL.AboRh _aboRH;
		// for UC100  (Justify ABORh), the ABO/Rh from the patient record is needed
		private BOL.AboRh _conversionABORh;
		private BOL.AboRh _previousPatientAboRH;
		
		private DataTable _clinicallySignificantAntibodies;
		private DataTable _dtSpecialInstructions;
		private DataTable _dtClinicallySignificantAntibodies;
		private DataTable _dtTransfusionRequirements = null;

		private long _vistaID;			//CR 2431	changed to long to allow for the database change of the VistaPatientId to Bigint

		private bool _electronicCrossmatchIndicator;
		private bool _importedFromVista = false;
		private bool _justifiedAboRH;
		private bool _validPatientDOB;

		// CR2899 start: cached information for faster processing
		// NOTE: DO NOT access these directly.  Access them through the corresponding
		// property!!!  Or else!
		private DataTable _dtAntigenTestsCache = null;
		private DataTable _dtAntibodyTypesCache = null;
		private DataRow [] _drPatientAntigensCache = null;
		private DataTable _dtClinicallySignificantAntibodiesCache = null;
		// CR2899 end

		#endregion

		#region Binding Change Events
		/// <summary>
		/// Patient GUID changed handler
		/// </summary>
		public event EventHandler PatientGUIDChanged;
		/// <summary>
		/// DisplayName changed handler
		/// </summary>
		public event EventHandler DisplayNameChanged;
		/// <summary>
		/// Patient GUID changed event
		/// </summary>
		protected virtual void OnPatientGUIDChanged()
		{
			if(PatientGUIDChanged != null)
				PatientGUIDChanged(this, EventArgs.Empty);
		}

		/// <summary>
		/// DisplayName Changed event
		/// </summary>
		protected virtual void OnDisplayNameChanged()
		{
			if(DisplayNameChanged != null)
				DisplayNameChanged(this, EventArgs.Empty);
		}

		#endregion
		
		#region Constructors

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/11/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="581"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Initialized patient object</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="3148"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Patient empty class constructor
		/// </summary>
		public Patient()
		{
			this._aboRH = new BOL.AboRh();
			this._previousPatientAboRH = new BOL.AboRh();
			this.Clear();
			this.IsNew = true;
			
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/26/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2413"> 
		///		<ExpectedInput>DataRow containing Patient data</ExpectedInput>
		///		<ExpectedOutput>Patient object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="8442"> 
		///		<ExpectedInput>Datarow containing Patient data</ExpectedInput>
		///		<ExpectedOutput>Patient object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2444"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Creates Patient object and populates it with data from DataRow
		/// </summary>
		/// <param name="dr"></param>
		public Patient(DataRow dr)
		{
			this._aboRH = new BOL.AboRh();
			this._previousPatientAboRH = new BOL.AboRh();
			this.LoadFromDataRow(dr);
		}


		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/26/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3149"> 
		///		<ExpectedInput>DataRow containing Patient data, bool</ExpectedInput>
		///		<ExpectedOutput>Patient object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3150"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Patient
		/// </summary>
		/// <param name="dr"></param>
		/// <param name="importedFromVista"></param>
		public Patient(DataRow dr, bool importedFromVista)
		{
			this._aboRH = new BOL.AboRh();
			this._previousPatientAboRH = new BOL.AboRh();
			this.ImportedFromVista = importedFromVista;
			this.LoadFromDataRow(dr);
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/26/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3151"> 
		///		<ExpectedInput>Guid</ExpectedInput>
		///		<ExpectedOutput>Patient object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3152"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Constructor
		/// </summary>
		/// <param name="patientGuid"></param>
		public Patient(System.Guid patientGuid)
		{
			this._aboRH = new BOL.AboRh();
			this._previousPatientAboRH = new BOL.AboRh();
			System.Data.DataTable dt = 	DAL.Patient.GetPatientDetails(patientGuid); 
			this._dtTransfusionRequirements = null;
			if(dt.Rows.Count > 0)
			{
				this.LoadFromDataRow(dt.Rows[0]);
			}
		}

		#endregion

		#region Properties

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/11/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="582"> 
		///		<ExpectedInput>Guid</ExpectedInput>
		///		<ExpectedOutput>Guid</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1864"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets patient unique identifier.
		/// Setting fires a change event.
		/// </summary>
		public Guid PatientGuid
		{
			get
			{
				return this._patientGUID;
			}
			set
			{
				this._patientGUID = value;
				this._dtTransfusionRequirements = null;
				this.OnPatientGUIDChanged();
			}
		}
		
		///<Developers>
		///	<Developer>D. Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/8/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8170"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8171"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets patient merge group unique identifier.
		/// Used to keep track of record merges for a patient.
		/// </summary>
		public Guid PatientMergeGroup
		{
			get
			{
				return this._patientMergeGroup;
			}
			set
			{
				this._patientMergeGroup = value;
			}
		}
		
		///<Developers>
		///	<Developer>D. Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/8/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8172"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8173"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets final merge to patient unique identifier.
		/// Indicates final merge record (patient guid) for a patient.
		/// </summary>
		public Guid FinalMergeToPatient
		{
			get
			{
				return this._finalMergeToPatient;
			}
			set
			{
				this._finalMergeToPatient = value;
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/11/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="583"> 
		///		<ExpectedInput>Valid patient object</ExpectedInput>
		///		<ExpectedOutput>Patient last name, first name and middle initial properties, formatted for display</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="3153"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets patient name in display format
		/// </summary>
		public string DisplayName
		{
			get
			{
				return BuildDisplayName(this._firstName,this._middleInitial,this._lastName);
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/10/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3510"> 
		///		<ExpectedInput>Date</ExpectedInput>
		///		<ExpectedOutput>Date</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3511"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// PatientDobCode
		/// </summary>
		public string PatientDobCode
		{
			get
			{
				return this._patientDOBCode;
			}
			set
			{
				this._patientDOBCode = value;
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/10/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3512"> 
		///		<ExpectedInput>bool</ExpectedInput>
		///		<ExpectedOutput>bool</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3513"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// ValidPatientDOB
		/// </summary>
		public bool ValidPatientDOB
		{
			get
			{
				if(this._patientDOBCode == "B" || 
					this._patientDOBCode == "D" ||
					this._patientDOBCode == "M")
				{
					this._validPatientDOB = false;
				}
				else
				{
					this._validPatientDOB = true;
				}
				return this._validPatientDOB;
			}
		}


			

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/11/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1865"> 
		///		<ExpectedInput>Valid patient first name string (0-20 chars)</ExpectedInput>
		///		<ExpectedOutput>Updated patient object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1866"> 
		///		<ExpectedInput>Invalid patient first name string (21 chars)</ExpectedInput>
		///		<ExpectedOutput>Business Object Exception thrown</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets patient's first name.
		/// ** NOTE: Use DisplayName property instead of LastName, FirstName and MiddleName when displaying in GUI **
		/// </summary>
		public string FirstName
		{
			get
			{
				return this._firstName;
			}
			set
			{
				//CR 2606
				if (value.Trim().Length>30)
				{
					throw new BusinessObjectException(Common.StrRes.SysErrMsg.Common.PatientNameTooLong().ResString);
				}
				this._firstName = value;
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/11/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1867"> 
		///		<ExpectedInput>Valid patient last name string (0-20 chars)</ExpectedInput>
		///		<ExpectedOutput>Updated patient object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1868"> 
		///		<ExpectedInput>Invalid patient last name string (21 chars)</ExpectedInput>
		///		<ExpectedOutput>Business Object Exception thrown</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets patient's last name
		/// ** NOTE: Use DisplayName property instead of LastName, FirstName and MiddleName when displaying in GUI **
		/// </summary>
		public string LastName
		{
			get
			{
				return this._lastName;
			}
			set
			{
				//CR 2606
				if (value.Trim().Length>30)
				{
					throw new BusinessObjectException(Common.StrRes.SysErrMsg.Common.PatientNameTooLong().ResString);
				}
				this._lastName = value;
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/11/2002</CreationDate>
		///<TestCases>
		///<Case type="0" testid ="1869"> 
		///		<ExpectedInput>string</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1870"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets patient's middle initial
		/// ** NOTE: Use DisplayName property instead of LastName, FirstName and MiddleName when displaying in GUI **
		/// </summary>
		public string MiddleInitial
		{
			get
			{
				return this._middleInitial;
			}
			set
			{
				//CR 2606
				if (value.Trim().Length>30)
				{
					throw new BusinessObjectException(Common.StrRes.SysErrMsg.Common.PatientNameTooLong().ResString);
				}
				this._middleInitial = value;
			}
		}



		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/11/2002</CreationDate>
		///<TestCases>
		///<Case type="0" testid ="3154"> 
		///		<ExpectedInput>bool</ExpectedInput>
		///		<ExpectedOutput>bool</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3155"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// ImportedFromVista
		/// </summary>
		public bool ImportedFromVista
		{
			get
			{
				return this._importedFromVista;
			}
			set
			{
				this._importedFromVista = value;
			}
		}


		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/11/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1876"> 
		///		<ExpectedInput>Valid SSN string (0 or 9 chars)</ExpectedInput>
		///		<ExpectedOutput>Updated patient object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1877"> 
		///		<ExpectedInput>Invalid SSN string (10 chars)</ExpectedInput>
		///		<ExpectedOutput>Business Object Exception thrown</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets patient's social security number
		/// ** DO NOT USE THIS PROPERTY IN GUI - USE DisplayPatientID instead **
		/// </summary>
		public string SSN
		{
			get
			{
				return this._ssn;
			}
			set
			{
				if (value.Trim().Length>9)
				{
					string errorMessage = Common.StrRes.SysErrMsg.Common.CannotbeGreaterError(Common.Utility.ParseStringwithSpaces(
						MethodBase.GetCurrentMethod().Name, true), "9").ResString;
					throw new BusinessObjectException(errorMessage);
				}
				this._ssn = value;
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/11/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3156"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Common.ABORhSource</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3157"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Indicates where tbe ABO/Rh results are coming from
		/// </summary>
		public Common.ABORhSource ABORhSource
		{
			get
			{				
				DataRow dtRow = DAL.Patient.GetPatientABORh(_patientGUID);
				Common.ABO abo =	Common.Utility.GetAboFromString(dtRow[Common.DatabaseConstants.ArtificialColumnNames.ABO].ToString());
				Common.RH rh =		Common.Utility.GetRhFromString(dtRow[Common.DatabaseConstants.ArtificialColumnNames.Rh].ToString());
				BOL.AboRh aboRh = new AboRh(abo, rh);

				if (abo == Common.ABO.NA) 
				{
					_aboRhSource = Common.ABORhSource.None;
				}
				else
				{
					// If test date is null, the result comes from the patient file
					// because there are no specimen tests recorded or the specimen
					// tests do not agree with the patient file and have not been
					// justified.
					if (dtRow.IsNull(TABLE.SpecimenTest.TestDate))
					{
						if ( ((bool) dtRow[Common.DatabaseConstants.ArtificialColumnNames.Discrepancy]) == true)
						{
							_aboRhSource = Common.ABORhSource.ConversionDiscrepancy;
						}
						else
						{
							_aboRhSource = Common.ABORhSource.Conversion;
						}
					}
					else
					{
						if ( ((bool) dtRow[Common.DatabaseConstants.ArtificialColumnNames.Discrepancy]) == true)
						{
							_aboRhSource = Common.ABORhSource.SpecimenTestDiscrepancy;
						}
						else
						{
							_aboRhSource = Common.ABORhSource.SpecimenTest;
						}
					}
				}
				return _aboRhSource;
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/11/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3158"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3159"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Gets Patient ID (DOD prefix ,SSN, and Pseudo suffix concatenated) for display.
		/// </summary>
		public string DisplayPatientID
		{
			get
			{
				return this._displayPatientID;
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/11/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1878"> 
		///		<ExpectedInput>Valid date of birth (non-future date)</ExpectedInput>
		///		<ExpectedOutput>Updated patient object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1879"> 
		///		<ExpectedInput>Invalid date of birth (future date)</ExpectedInput>
		///		<ExpectedOutput>BusinessObjectException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets patient date of birth
		/// </summary>
		public DateTime DateOfBirth
		{
			get
			{
				return this._dob;
			}
			set
			{
				if (value > VBECSDateTime.GetDivisionCurrentDateTime())
				{
					string errorMessage = Common.StrRes.SysErrMsg.Common.CannotbeGreaterError(Common.Utility.ParseStringwithSpaces(
						MethodBase.GetCurrentMethod().Name, true), "today's date").ResString;
					throw new BOL.BusinessObjectException(errorMessage);
				}
				
				this._dob = value;
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/11/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1978"> 
		///		<ExpectedInput>Valid specimen UID string (0-20 chars)</ExpectedInput>
		///		<ExpectedOutput>Updated patient objct</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1979"> 
		///		<ExpectedInput>Invalid specimen UID string (21 chars)</ExpectedInput>
		///		<ExpectedOutput>Business Object Exception thrown</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets specimen UID of current patient order
		/// </summary>
		public string SpecimenUid
		{
			get
			{
				return this._specimenUid;
			}
			set
			{
				if (value.Trim().Length>20)
				{
					string errorMessage = Common.StrRes.SysErrMsg.Common.CannotbeGreaterError(Common.Utility.ParseStringwithSpaces(
						MethodBase.GetCurrentMethod().Name, true), "20").ResString;
					throw new BusinessObjectException(errorMessage);
				}
				this._specimenUid = value;
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/11/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="591"> 
		///		<ExpectedInput>Common.Sex</ExpectedInput>
		///		<ExpectedOutput>Common.Sex</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1982"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets gender of patient
		/// </summary>
		public Common.Sex Sex
		{
			get
			{
				return this._sex;
			}
			set
			{
				this._sex = value;
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/12/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7551"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7552"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// HasActiveSIs
		/// </summary>
		public bool HasActiveSIs
		{
			get
			{
				for (int i = 0; i < this.SpecialInstructions.Rows.Count; i++)
				{
					Common.RecordStatusCode status = Common.Utility.GetRecordStatusCodeFromString(SpecialInstructions.Rows[i][TABLE.PatientSpecialInstruction.RecordStatusCode].ToString());
					if ( status == Common.RecordStatusCode.Active || status == Common.RecordStatusCode.Converted )
					{
						return true;
					}
				}
				return false;
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/12/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7553"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="9163"> 
		///		<ExpectedInput>TRs from conversion</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7554"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// HasActiveTRs
		/// </summary>
		public bool HasActiveTRs
		{
			get
			{
				int nCount = this.TransfusionRequirements.Rows.Count;
				for (int i = 0; i < nCount ; i++)
				{
					Common.RecordStatusCode status = Common.Utility.GetRecordStatusCodeFromString(TransfusionRequirements.Rows[i][TABLE.PatientTransfusionRequirement.RecordStatusCode].ToString());
					if ( status == Common.RecordStatusCode.Active || status == Common.RecordStatusCode.Converted )
					{
						return true;
					}
				}
				return false;
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/10/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3514"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3515"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// SpecialInstructions
		/// BR_
		/// </summary>
		public DataTable SpecialInstructions
		{
			get
			{
				if (this._dtSpecialInstructions == null)
				{
					this.LoadSpecialInstructions();
				}
				return this._dtSpecialInstructions;
			}
		}
		
		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/10/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3516"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3517"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// TransfusionRequirements - This DataTable holds TRs that are both Active and Inactive.  T
		/// his is to accomodate the Maintain Transfusion Requirements UC.
		/// BR_38.08
		/// </summary>
		public DataTable TransfusionRequirements
		{
			get
			{
				this.LoadTransfusionRequirements();
				return _dtTransfusionRequirements;
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/10/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3518"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="9164"> 
		///		<ExpectedInput>Patient with TRs from conversion</ExpectedInput>
		///		<ExpectedOutput>DataTable containing TRs from conversion</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3519"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// ClinicallySignificantAntibodies
		/// </summary>
		public DataTable ClinicallySignificantAntibodies
		{
			get
			{
				_clinicallySignificantAntibodies = LoadClinicallySignificantAntibodies();
				return _clinicallySignificantAntibodies;
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/11/2002</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="1987"> 
		///		<ExpectedInput>Valid VistA patient ID (greater than 0)</ExpectedInput>
		///		<ExpectedOutput>Updated patient object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1988"> 
		///		<ExpectedInput>Invalid VistA patient ID (less than 1)</ExpectedInput>
		///		<ExpectedOutput>Business Object Exception thrown</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets the ID in VistA of patient
		/// CR 2431	changed to long to allow for the database change of the VistaPatientId to Bigint
		/// </summary>
		public long VistaPatientId
		{
			get
			{
				return this._vistaID;
			}
			set
			{
				if ((value < 1) && (value != long.MinValue))
				{
					string errorMessage = Common.StrRes.SysErrMsg.Common.InvalidPropertyOrMethod(
						Common.Utility.ParseStringwithSpaces(MethodBase.GetCurrentMethod().Name, true),
						value).ResString;
					throw new BusinessObjectException(errorMessage);
				}
				this._vistaID = value;
			}
		}

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/31/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6089"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6090"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// ToDo: Complete Unit tests
		/// BR_36.05, BR_36.06
		/// </summary>
		public string CompatibilityPercentage
		{
			get
			{
				return GetCompatibilityPercentage();
			}
		}

		
		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/10/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3162"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>BOL.AboRh</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3163"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// The current patient ABORh
		/// </summary>
		public BOL.AboRh AboRh
		{
			get
			{
				return this._aboRH;
			}
			set
			{
				this._aboRH = value;
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/10/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3164"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>BOL.AboRh</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3165"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// ABORh from patient conversion
		/// </summary>
		public BOL.AboRh ConversionAboRH
		{
			get
			{
				return _conversionABORh;
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/10/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3166"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>BOL.AboRh</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3171"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Gets the result of previous AboRh testing for patient.
		/// If there is no previous testing the conversion AboRh is returned.
		/// </summary>
		public BOL.AboRh PreviousAboRH
		{
			get
			{
				if (this._previousPatientAboRH.Abo == Common.ABO.NA &&
					this._previousPatientAboRH.RH == Common.RH.NotProvided)
				{
					DataTable dtPatientRecentAboRhTesting = DAL.Patient.GetRecentPatientAboRHHistory(this.PatientGuid);
					if (dtPatientRecentAboRhTesting.Rows.Count < 4)
					{
						return this.ConversionAboRH;
					}
					else
					{
						DataRow[] drAboRhTesting = dtPatientRecentAboRhTesting.Select("", VbecsTables.SpecimenTest.TestDate + " ASC, " + VbecsTables.RhFactor.RhFactorText);
						string Abo = drAboRhTesting[0][VbecsTables.SpecimenTest.TestResultId].ToString().Trim();
						string Rh = drAboRhTesting[1][VbecsTables.SpecimenTest.TestResultId].ToString().Trim();

						return new BOL.AboRh(Common.Utility.GetAboFromString(Abo), Common.Utility.GetRhFromString(Rh));
					}
				}
				else
				{
					return this._previousPatientAboRH;
				}
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/10/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3172"> 
		///		<ExpectedInput>string</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3173"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// PatientServiceBranch
		/// </summary>
		public string PatientServiceBranch
		{
			get
			{
				return this._patientServiceBranch;
			}
			set
			{
				this._patientServiceBranch = value;
			}
		}
        
		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/10/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3174"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>bool</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3175"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// ValidSpecimentExists
		/// </summary>
		public bool ValidSpecimenExists
		{
			get 
			{
				return (DAL.PatientSpecimen.ValidSpecimenExists(this._patientGUID).Rows.Count > 0);
			}
		}


		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/10/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3176"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>bool</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3177"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// ElectronicCrossmatchIndicator - Be very careful when using this.  
		/// this.DetermineElectronicCrossmatchEligibility() must be called first to 
		/// set this property.  ToDo: Refactor!
		/// </summary>
		public bool ElectronicCrossmatchIndicator 
		{
			get 
			{
				return this._electronicCrossmatchIndicator;
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/10/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3178"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3180"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// ABORhHistory
		/// </summary>
		public DataTable AboRHHistory
		{
			get 
			{
				return GetPatientAboRHHistory();
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/10/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3183"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>bool</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3184"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Inidcates if patient has a history of Justified ABO/Rh changes
		/// </summary>
		public bool JustifiedAboRH
		{
			get
			{
				return this._justifiedAboRH;
			}
		}

		

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/12/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7555"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7556"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// IsPatientDeceased
		/// </summary>
		public bool IsPatientDeceased
		{
			get
			{
				return (this._patientDeathDate != DateTime.MinValue);
			}
		}

		#endregion



		#region Cache Properties

		private DataTable AntigenTestsCache
		{
			get
			{
				if 	(_dtAntigenTestsCache == null)
				{
					_dtAntigenTestsCache = BOL.AntibodyType.GetAntigenTest();
				}
				return _dtAntigenTestsCache;
			}
		}
		

		private DataTable AntibodyTypesCache
		{
			get
			{
				if 	(_dtAntibodyTypesCache == null)
				{
					_dtAntibodyTypesCache = DAL.ReferenceData.GetAntibodyTypeList();
				}
				return _dtAntibodyTypesCache;
			}
		}


		private DataRow [] PatientAntigensCache
		{
			get
			{
				if 	(_drPatientAntigensCache == null)
				{
					_drPatientAntigensCache = this.TransfusionRequirements.Select(VbecsTables.PatientTransfusionRequirement.TransfusionRequirementCategoryCode + " = '" + Common.Utility.GetTransfusionCategoryCodeFromEnum(Common.TransfusionRequirementCategoryCode.AntigenNegative) + "' AND RecordStatusCode = '" + Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Active) + "'");
				}
				return _drPatientAntigensCache;
			}
		}


		// This is only to be used for performance challenged UCs.  
		// ClinicallySignificantAntibodies should be used otherwise.
		private DataTable ClinicallySignificantAntibodiesCache
		{
			get
			{
				if 	(_dtClinicallySignificantAntibodiesCache  == null)
				{
					_dtClinicallySignificantAntibodiesCache = this.ClinicallySignificantAntibodies;
				}
				return _dtClinicallySignificantAntibodiesCache;
			}
		}


		
		#endregion



		#region Methods


        ///<Developers>
        ///	<Developer>Carl Jensen</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>5/12/2015</CreationDate>
        /// <summary>
        /// Check to make sure transfusion requirements didn't change in between the time the unit was selected and when the user clicked save.
        /// This was added as part of the fix for CR 3550.
        /// </summary>
        public bool DidAntigenNegativeRequirementsChange()
        {
            // First, get refreshed antigens and antibodies
            DataRow[] refreshedAntigens = this.TransfusionRequirements.Select(VbecsTables.PatientTransfusionRequirement.TransfusionRequirementCategoryCode + " = '" + Common.Utility.GetTransfusionCategoryCodeFromEnum(Common.TransfusionRequirementCategoryCode.AntigenNegative) + "' AND RecordStatusCode = '" + Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Active) + "'");
            DataTable refreshedAntibodies = LoadClinicallySignificantAntibodies();

            if (refreshedAntigens.Length != PatientAntigensCache.Length || refreshedAntibodies.Rows.Count != ClinicallySignificantAntibodiesCache.Rows.Count) return true;

            Boolean found = false;

            // Now, compare to the cached ones
            foreach (DataRow dtRowRefreshedAntigen in refreshedAntigens)
            {
                int antigenTypeIdRefreshed = (int)dtRowRefreshedAntigen[TABLE.PatientTransfusionRequirement.AntigenTypeId];

                // Finally, make sure that this antigen is in the cached table
                foreach (DataRow dtRowCachedAntigen in PatientAntigensCache)
                {
                    found = false;
                    if (antigenTypeIdRefreshed == (int)dtRowCachedAntigen[TABLE.PatientTransfusionRequirement.AntigenTypeId])
                    {
                        found = true;
                        break;
                    }
                }
                if (!found) return true;
            }

            foreach (DataRow dtRowRefreshedAntibody in refreshedAntibodies.Rows)
            {
                int antibodyTypeIdRefreshed = (int)dtRowRefreshedAntibody[VbecsTables.PatientTransfusionRequirement.AntibodyTypeId];

                foreach (DataRow dtRowCachedAntibody in ClinicallySignificantAntibodiesCache.Rows)
                {
                    found = false;
                    if (antibodyTypeIdRefreshed == (int)dtRowCachedAntibody[VbecsTables.PatientTransfusionRequirement.AntibodyTypeId])
                    {
                        found = true;
                        break;
                    }
                }
                if (!found) return true;
            }
            return false;
        }


        ///<Developers>
        ///	<Developer>Carl Jensen</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>5/12/2015</CreationDate>
        /// <summary>
        /// Check to make sure transfusion requirements didn't change in between the time the unit was selected and when the user clicked save.
        /// This was added as part of the fix for CR 3550.
        /// </summary>
        /// <param name="ComponentRequirementCache"></param>
        /// <param name="AllUnits"></param>
        public bool DidTransfusionRequirementsChange(Hashtable ComponentRequirementCache, ArrayList AllUnits)
        {
            Hashtable refreshedComponentRequirements = this.ValidateComponentRequirements(AllUnits);

            if (refreshedComponentRequirements.Count != ComponentRequirementCache.Count) return true;

            IDictionaryEnumerator enumCached = ComponentRequirementCache.GetEnumerator();

            while (enumCached.MoveNext())
            {
                Guid cachedGuid = (Guid)enumCached.Key;
                ArrayList cachedReqs = (ArrayList)enumCached.Value;

                ArrayList refreshedReqs = (ArrayList)refreshedComponentRequirements[cachedGuid];

                if (cachedReqs.Count != refreshedReqs.Count) return true;

                foreach (string requirement in cachedReqs)
                {
                    if (!refreshedReqs.Contains(requirement)) return true;
                }

            }
            return false;
        }





		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/11/2002</CreationDate>
		/// <summary>
		/// Method to load properties of patient object from a supplied data row
		/// Implements BR_56.10
		/// </summary>
		/// <param name="dtRow"></param>
		protected override void LoadFromDataRow(System.Data.DataRow dtRow)
		{
			if (dtRow.Table.Columns.Contains(VbecsTables.Patient.PatientGuid))
			{
				this.PatientGuid = new Guid(dtRow[VbecsTables.Patient.PatientGuid].ToString());
			}
			if (dtRow.Table.Columns.Contains(VbecsTables.Patient.PatientFirstName))
			{
				this.FirstName = dtRow[VbecsTables.Patient.PatientFirstName].ToString();
			}
			if (dtRow.Table.Columns.Contains(VbecsTables.Patient.PatientMiddleName))
			{
				this.MiddleInitial = dtRow[VbecsTables.Patient.PatientMiddleName].ToString();
			}
			if (dtRow.Table.Columns.Contains(VbecsTables.Patient.PatientLastName))
			{
				this.LastName  = dtRow[VbecsTables.Patient.PatientLastName].ToString();
			}
			if (dtRow.Table.Columns.Contains(VbecsTables.Patient.PatientSsn))
			{
				this.SSN = dtRow[VbecsTables.Patient.PatientSsn].ToString();
			}
			if (dtRow.Table.Columns.Contains(VbecsTables.Patient.PatientDob))
			{
				if (!dtRow.IsNull(VbecsTables.Patient.PatientDob))
				{
					try
					{
						this.DateOfBirth = (System.DateTime) dtRow[VbecsTables.Patient.PatientDob];
					}
					catch(System.InvalidCastException)
					{
						this.DateOfBirth = Common.HL7DateFormat.ConvertHL7DateTime(dtRow[VbecsTables.Patient.PatientDob].ToString());
					}
				}
			}
			
			_conversionABORh = new AboRh();
			if (dtRow.Table.Columns.Contains(Common.DatabaseConstants.ArtificialColumnNames.ConversionBloodTypeCode))
			{
				_conversionABORh.Abo = BOL.AboRh.GetAboFromString(dtRow[Common.DatabaseConstants.ArtificialColumnNames.ConversionBloodTypeCode].ToString().Trim());
				try
				{
					_conversionABORh.RH = BOL.AboRh.GetRHFromCode(dtRow[Common.DatabaseConstants.ArtificialColumnNames.ConversionRhFactorCode].ToString().Trim());
				}
				catch(BOL.BusinessObjectException)
				{
					_conversionABORh.RH = Common.RH.NotProvided;
				}
			}

			// BR_56.10
			if (dtRow.Table.Columns.Contains(VbecsTables.Patient.BloodTypeCode))
			{
				if(dtRow[VbecsTables.Patient.BloodTypeCode].ToString() != String.Empty)
				{
					_aboRH = new AboRh(Common.Utility.GetAboFromString(dtRow[VbecsTables.Patient.BloodTypeCode].ToString()), Common.Utility.GetRhFromString(dtRow[VbecsTables.Patient.RhFactorCode].ToString()));
				}
			}

			//	
			if (dtRow.Table.Columns.Contains(VbecsTables.Patient.PatientSexCode))
			{
				if(dtRow[VbecsTables.Patient.PatientSexCode].ToString() == String.Empty)
				{
					this._sex = Common.Sex.Unknown;
				}
				else
				{
					this._sex = dtRow.IsNull(VbecsTables.Patient.PatientSexCode) ? Common.Sex.Unknown : Common.Utility.GetSexEnumFromSexChar(System.Convert.ToChar(dtRow[VbecsTables.Patient.PatientSexCode]));
				}
			}
			//CR 2431	changed to Int64 to allow for the database change of the VistaPatientId to Bigint
			if (dtRow.Table.Columns.Contains(VbecsTables.Patient.VistaPatientId))
			{
				this.VistaPatientId = dtRow.IsNull(VbecsTables.Patient.VistaPatientId) ? Int64.MinValue : System.Convert.ToInt64(dtRow[VbecsTables.Patient.VistaPatientId]);
			}	

			if (dtRow.Table.Columns.Contains(VbecsTables.Patient.PatientServiceBranch))
			{
				this.PatientServiceBranch = dtRow.IsNull(VbecsTables.Patient.PatientServiceBranch) ? string.Empty : dtRow[VbecsTables.Patient.PatientServiceBranch].ToString();
			}
			if (dtRow.Table.Columns.Contains(VbecsTables.Patient.PatientDobCode))
			{
				this.PatientDobCode = dtRow.IsNull(VbecsTables.Patient.PatientDobCode) ? string.Empty : dtRow[VbecsTables.Patient.PatientDobCode].ToString();
			}
			if (dtRow.Table.Columns.Contains(VbecsTables.Patient.DisplayVistaPatientId))
			{
				this._displayPatientID = dtRow[VbecsTables.Patient.DisplayVistaPatientId].ToString();
			}
			if (dtRow.Table.Columns.Contains(VbecsTables.Patient.RowVersion) && !dtRow.IsNull(VbecsTables.Patient.RowVersion))
			{
				this.RowVersion = (byte [])dtRow[VbecsTables.Patient.RowVersion];
			}
			if (dtRow.Table.Columns.Contains(VbecsTables.Patient.AboRhChangeIndicator))
			{
				this._justifiedAboRH = dtRow.IsNull(VbecsTables.Patient.AboRhChangeIndicator) ? false : (bool)dtRow[VbecsTables.Patient.AboRhChangeIndicator];
			}
			if (dtRow.Table.Columns.Contains(VbecsTables.Patient.PatientDeathDate))
			{
				if (!dtRow.IsNull(VbecsTables.Patient.PatientDeathDate))
				{
					this._patientDeathDate = Convert.ToDateTime(dtRow[VbecsTables.Patient.PatientDeathDate]);
				}
				else
				{
					this._patientDeathDate = DateTime.MinValue;
				}
			}
			if (dtRow.Table.Columns.Contains(VbecsTables.Patient.FinalMergeToPatient))
			{
				if (!dtRow.IsNull(VbecsTables.Patient.FinalMergeToPatient))
				{
					this._finalMergeToPatient = (Guid)(dtRow[VbecsTables.Patient.FinalMergeToPatient]);
				}
			}
			if (dtRow.Table.Columns.Contains(VbecsTables.Patient.PatientMergeGroup))
			{
				if (!dtRow.IsNull(VbecsTables.Patient.PatientMergeGroup))
				{
					this._patientMergeGroup = (Guid)(dtRow[VbecsTables.Patient.PatientMergeGroup]);
				}
			}
		}

		
		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/10/2003</CreationDate>
		/// <summary>
		/// Determine the eligibility of patient for electronic crossmatch
		/// Implements BR_3.17, BR_3.18
		/// 
		/// UnitTestNote: This method is completely tested by PatientUnitSelection unit tests.  There is no
		/// need to add any here.
		/// </summary>
		/// <param name="specimen"></param>
		/// <param name="divisionCode"></param>
		/// <returns></returns>
		public string DetermineElectronicCrossmatchEligibility(BOL.Specimen specimen, string divisionCode)
		{
			_electronicCrossmatchIndicator = false;

			if (specimen == null || specimen.IsNew)
			{
				return Common.StrRes.InfoMsg.Common.PatienteXMDisabledNoSpecimen().ResString;
			}

			// TT_3.05,1: Current specimen ABO/Rh and Antibody Screen testing must be complete.
			DataTable dtTestResult = specimen.TestResults;
			DataRow[] drABO		= dtTestResult.Select(TABLE.SpecimenTest.BloodTestTypeId + " = " + (int) Common.TestType.ABOInterp + " OR " + TABLE.SpecimenTest.BloodTestTypeId + " = " + (int) Common.TestType.ABOInterpTAS);
			DataRow[] drRh		= dtTestResult.Select(TABLE.SpecimenTest.BloodTestTypeId + " = " + (int) Common.TestType.RhInterp + " OR " + TABLE.SpecimenTest.BloodTestTypeId + " = " + (int) Common.TestType.RhInterpTAS);
			DataRow[] drAntibody = dtTestResult.Select(TABLE.SpecimenTest.BloodTestTypeId + " = " + (int) Common.TestType.AntibodyScreen + " OR " + TABLE.SpecimenTest.BloodTestTypeId + " = " + (int) Common.TestType.AntibodyScreenTAS);
			
			bool resultsExist = false;
			
			// first, check abo results
			for (int i=0; i<drABO.Length; i++) 
			{
				if (drABO[i][TABLE.SpecimenTest.TestResultId] != System.DBNull.Value) 
				{
					if (resultsExist) break;

					// next, check rh results
					for (int j=0; j<drRh.Length; j++) 
					{
						if (resultsExist) break;

						if (drRh[j][TABLE.SpecimenTest.TestResultId] != System.DBNull.Value) 
						{
							// finally, check antibody results
							for (int k=0; k<drAntibody.Length; k++) 
							{
								if (drAntibody[k][TABLE.SpecimenTest.TestResultId] != System.DBNull.Value) 
								{
									resultsExist = true;
									break;
								}
							}
						}
					}
				}
			}
			if (!resultsExist)
			{
				return Common.StrRes.InfoMsg.Common.PatienteXMDisabledNoSpecimenTesting().ResString;
			}

			// TT_3.05, 2: Current specimen Antibody Screen interpretation must be negative.
			foreach (DataRow dr in drAntibody)
			{	
				string testResultID = dr[TABLE.SpecimenTest.TestResultId].ToString().Trim();
				if (!testResultID.Equals("N"))
				{
					return Common.StrRes.InfoMsg.Common.PatienteXMDisabledPositiveAntibody().ResString;
				}
			}
 
			// TT_3.05, 3: Patient must have no history of previously identified antibodies, regardless of clinical 
			// significance based on PT_23.01 or division (or from conversion) of the antibody ID.	
			DataRow[] drAntibodies = this.TransfusionRequirements.Select(TABLE.PatientTransfusionRequirement.TransfusionRequirementCategoryCode + " = 'AS'");
			foreach(DataRow dr in drAntibodies)
			{
				if (dr[TABLE.PatientTransfusionRequirement.RecordStatusCode].ToString().Equals(Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Active).ToString()) ||
					// CR2502
					dr[TABLE.PatientTransfusionRequirement.RecordStatusCode].ToString().Equals(Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Converted).ToString()))
					// CR2502 end
				{
					return Common.StrRes.InfoMsg.Common.PatienteXMDisabledAntibodyHistory().ResString;
				}
			}
			
			// TT_3.05, 4: Patient cannot have a persistent antigen negative requirement regardless of the division of the requirement entry.	.
			DataRow[] drAntigens = this.TransfusionRequirements.Select(TABLE.PatientTransfusionRequirement.TransfusionRequirementCategoryCode + "  = 'AN'");
			foreach(DataRow dr in drAntigens)
			{
				if (dr[TABLE.PatientTransfusionRequirement.RecordStatusCode].ToString().Equals(Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Active).ToString()))
				{
					return Common.StrRes.InfoMsg.Common.PatienteXMDisabledAntigenNegative().ResString;
				}
			}
	
			int aboTests = 0;
			int rhTests = 0;
			foreach (DataRow dr in this.AboRHHistory.Rows)
			{
				int test = (int) dr[TABLE.BloodTestType.BloodTestTypeId];
				if ( test == (int) Common.TestType.ABOInterp || test == (int) Common.TestType.ABORepeat || test == (int) Common.TestType.ABOInterpTAS)
				{
					// TT_3.05, 6: The patient cannot have a documented instance of current or previous serologic problems, such as a valid Inconclusive ABO/Rh typing.
					if (Common.ABO.I == AboRh.GetAboFromString(dr[TABLE.SpecimenTest.TestResultId].ToString().Trim()) )
					{
						return Common.StrRes.InfoMsg.Common.PatienteXMDisabledTypingDifficulty().ResString;
					}
						// TT_3.05, 5: The patient current ABO/Rh must match the historical record when there is a historical record and there must be no previous 
						// justifications. An ABO/Rh marked as entered in error is not considered part of the patients record.
					else if (_aboRH.Abo != AboRh.GetAboFromString(dr[TABLE.SpecimenTest.TestResultId].ToString()) )
					{
						return Common.StrRes.InfoMsg.Common.PatienteXMDisabledABORhDiscrepancy().ResString;
					}
					else if ((bool) dr[TABLE.SpecimenTest.AboRhChangeIndicator])
					{
						return Common.StrRes.InfoMsg.Common.PatienteXMDisabledABORhDiscrepancy().ResString;
					}
					else if (divisionCode.Trim().Equals( dr[TABLE.SpecimenTest.DivisionCode].ToString().Trim() ))
					{
						aboTests++;
					}
				}
				else if (test == (int) Common.TestType.RhInterp || test == (int) Common.TestType.RhRepeat || test == (int) Common.TestType.RhInterpTAS)
				{
					// TT_3.05, 6: The patient cannot have a documented instance of current or previous serologic problems, such as a valid Inconclusive ABO/Rh typing.
					if (Common.RH.Inconclusive == AboRh.GetRHFromCode(dr[TABLE.SpecimenTest.TestResultId].ToString()) )
					{
						return Common.StrRes.InfoMsg.Common.PatienteXMDisabledTypingDifficulty().ResString;
					}
						// TT_3.05, 5: The patient current ABO/Rh must match the historical record when there is a 
						// historical record. An ABO/Rh marked as entered in erroris not considered part of the 
						// patients record.
					else if (_aboRH.RH != AboRh.GetRHFromCode(dr[TABLE.SpecimenTest.TestResultId].ToString()) )
					{
						return Common.StrRes.InfoMsg.Common.PatienteXMDisabledABORhDiscrepancy().ResString;
					}
					else if (divisionCode.Trim().Equals( dr[TABLE.SpecimenTest.DivisionCode].ToString().Trim() ))
					{
						rhTests++;
					}
				}
			}

			// TT_3.05, 7: There must be at least two valid instances of ABO/Rh typing performed in the same division 
			// for the patient.  This means that there must be at least 4 test results (2 ABO & 2 Rh).
			// DataRow[] drABORhHistory = this.AboRHHistory.Select("DivisionCode = '" + this.DivisionCode + "'");
			if (aboTests < 2 || rhTests < 2)
			{
				return Common.StrRes.InfoMsg.Common.PatienteXMDisabledNeedTwoTypings().ResString;
			}			
			
			_electronicCrossmatchIndicator = true;
			return Common.StrRes.InfoMsg.Common.PatienteXMEnabled().ResString;
		}


		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/10/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3524"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3525"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Implements BR_3.05, BR_56.10,BR_65.03
		/// </summary>
		/// /// ABO/Rh actually consists of two different tests: an ABO interpretation and an Rh interpretation
		public String GetPatientHistoricABORhForTooltip()
		{
			DataRow dtRow = DAL.Patient.GetPatientABORh(_patientGUID);
		
			Common.ABO abo =	Common.Utility.GetAboFromString(dtRow[Common.DatabaseConstants.ArtificialColumnNames.ABO].ToString());
			Common.RH rh =		Common.Utility.GetRhFromString(dtRow[Common.DatabaseConstants.ArtificialColumnNames.Rh].ToString());
			BOL.AboRh aboRh = new AboRh(abo, rh);

			// No specimen tests and no ABO/Rh in the Patient Record
			if (abo == Common.ABO.NA) 
			{
				return Common.StrRes.InfoMsg.Common.NoABORhResults().ResString;
			}
			else
			{
				// Get ABO
				string aboString = Common.Utility.GetAboFromEnum(abo);
				// Get Rh
				string rhString = string.Empty;
				if (rh == Common.RH.Positive) rhString = "Pos";
				else if (rh == Common.RH.Negative) rhString = "Neg";
				else rhString = Common.Utility.GetRhFromEnum(rh);
				
				// ABO is from patient record if no test date
				if (dtRow[TABLE.SpecimenTest.TestDate] == System.DBNull.Value)
				{
					return "From conversion";
				}
				else
				{
					string testDate = Common.VBECSDateTime.FormatDateTimeString( (DateTime) dtRow[TABLE.SpecimenTest.TestDate]);
					string testDivision = dtRow[Common.DatabaseConstants.ArtificialColumnNames.TestDivision].ToString();
					return aboString + " " + rhString + ", Date: " + testDate + ", Division: " + testDivision;
				}
			}
		}


		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/14/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1233"> 
		///		<ExpectedInput>DataRow</ExpectedInput>
		///		<ExpectedOutput>DataRow</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="3861"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		///  Loads a data row from an object
		/// </summary>
		public override System.Data.DataRow LoadDataRowFromThis(System.Data.DataRow dtRow)
		{
			return dtRow;
		}



		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/11/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="596"> 
		///		<ExpectedInput>Initialized patient object</ExpectedInput>
		///		<ExpectedOutput>Updated patient object</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="3862"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Public virtual method to reset public properties of Patient object.
		/// </summary>
		public virtual void Clear()
		{
			this.PatientGuid = Guid.Empty;
			this.LastName = string.Empty;
			this.FirstName = string.Empty;
			this.MiddleInitial = string.Empty;
			this.AboRh.Abo = Common.ABO.NA;
			this.AboRh.RH = Common.RH.NotProvided;
			this.SSN = string.Empty;
			this.Sex = Common.Sex.Unknown;
			this.VistaPatientId = long.MinValue;		//CR 2431	changed to long to allow for the database change of the VistaPatientId to Bigint
			this._displayPatientID = string.Empty;
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/19/2003</CreationDate>
		/// <summary>
		/// Static funcion calls data access layer to return a data table containing
		/// special instructions for a supplied patient within a given division
		/// BR_36.01
		/// </summary>
		private void LoadSpecialInstructions()
		{
			this._dtSpecialInstructions = BOL.PatientSpecialInstruction.GetSpecialInstructionsForPatient(this._patientGUID);
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/17/2003</CreationDate>
		/// <summary>
		/// Load transfusion requirements for a patient
		/// BR_38.11
		/// </summary>
		private void LoadTransfusionRequirements()
		{
			_dtTransfusionRequirements = BOL.PatientTransfusionRequirement.GetTransfusionRequirementsForPatient(_patientGUID);
		}


		/// <summary>
		/// This method loads the antibodies that are clinically significant.
		/// </summary>
		/// <returns></returns>
		private DataTable LoadClinicallySignificantAntibodies()
		{
			int nCount = 0;
			DataTable dtTRs = BOL.PatientTransfusionRequirement.GetTransfusionRequirementsForPatient(_patientGUID);
			// filter to only get Antibody Requirements
			DataRow [] drAntibodiesArray = dtTRs.Select(TABLE.PatientTransfusionRequirement.TransfusionRequirementCategoryCode + " = '" + 
				Common.Utility.GetTransfusionCategoryCodeFromEnum(Common.TransfusionRequirementCategoryCode.AntibodyIdentified) + "' AND RecordStatusCode IN ('" + 
				Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Active) + "','" + Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Converted) + "')");

			_dtClinicallySignificantAntibodies = dtTRs.Clone();

			DataTable dtAntibodyParameters = BOL.AntibodyType.GetAntibodyTypeLookupList();
			if (dtAntibodyParameters.Rows.Count == 0)
			{
				throw new Exception("Antibodies have not been configured for this division!");
			}
			else
			{
				nCount = drAntibodiesArray.Length;
				for(int i=0; i<nCount; i++)
				{
					DataRow [] drAntibodies = dtAntibodyParameters.Select(TABLE.AntibodyParameter.AntibodyTypeId + " = " 
						+ (int) drAntibodiesArray[i][TABLE.AntibodyParameter.AntibodyTypeId]);
					if ( drAntibodies.Length > 0 && ((bool) drAntibodies[0][TABLE.AntibodyParameter.HigherLevelOverrideIndicator]) == true )
					{
						System.Data.DataRow dr = _dtClinicallySignificantAntibodies.NewRow();
						dr.ItemArray = drAntibodiesArray[i].ItemArray;
						_dtClinicallySignificantAntibodies.Rows.Add(dr);
					}
				}
			}
			return this._dtClinicallySignificantAntibodies;

		}


		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/10/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3530"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3531"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns a data table containing patient's ABO/Rh history from the patient GUID
		/// </summary>
		/// <returns></returns>
		public DataTable GetPatientAboRHHistory()
		{
			return DAL.Patient.GetPatientAboRHHistory(this.PatientGuid);
		}



		// CR2899 start

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/13/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5798"> 
		///		<ExpectedInput>Valid bloodunit</ExpectedInput>
		///		<ExpectedOutput>Arraylist</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5799"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// This method returns an ArrayList of BloodUnitAntigens indicating whether or not the 
		/// unit has been tested for the Antigen.  If it has been tested, the test result is also 
		/// indicated. 
		/// Implements BR_3.06, BR_3.15, BR_43.18, BR_56.22, BR_73.14
		/// </summary>
		/// <param name="bloodUnit"></param>
		/// <returns></returns>
		public ArrayList ValidateAntibodyAntigenRequirementsMet(BOL.BloodUnit bloodUnit)
		{
			ArrayList units = new ArrayList(1);
			units.Add(bloodUnit);

			return (ArrayList) (ValidateAntibodyAntigenRequirementsMet(units))[bloodUnit.BloodUnitGuid];
		}

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/11/2010</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="9155"> 
		///		<ExpectedInput>ArrayList of units</ExpectedInput>
		///		<ExpectedOutput>Hashtable of unit results</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="9167"> 
		///		<ExpectedInput>Empty ArrayList</ExpectedInput>
		///		<ExpectedOutput>ArgumentExcepton</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// This method returns a Hashtable of BloodUnitAntigens indicating whether or not the 
		/// unit has been tested for the Antigen.  If it has been tested, the test result is also 
		/// indicated. 
		/// Implements BR_3.06, BR_3.15, BR_43.18, BR_56.22, BR_73.14
		/// </summary>
		/// <param name="units"></param>
		/// <returns></returns>
		/// Developer note (12/17/2010): This code is way too complicated.  I have added a bunch 
		/// of comments and entered CR2916 to address the complexity issue.
		public Hashtable ValidateAntibodyAntigenRequirementsMet(ArrayList units)
		{
			if (units == null || units.Count == 0)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidArgument("units").ResString);
			}
			// This is the main hashtable that will be returned at the end of this monstrosity.  It 
			// holds the bloodUnitAntigens object below.
			Hashtable result = new Hashtable();

			// This is where results for units will be stored.  So, if 3 units are passed in, 3 
			// bloodunitantigens will be passed out.
			ArrayList bloodUnitAntigens = new ArrayList();

			// This one keeps track of requirements that have already been added.  We don't want to 
			// add the same thing twice.
			// Refactoring note: consider doing away with this and just checking the bloodUnitAntigens
			// object.
			ArrayList alreadyAdded = new ArrayList();

			// Keeps track on antigen tests.
			// Refactoring note: give this a better name.
			int nCount = 0;
						

			// Put the unit guids in a comma delimited string for the sproc calls 
			// that come next.
			StringBuilder sbUnits = new StringBuilder();
			for (int i=0; i<units.Count; i++)
			{
				if (i > 0) sbUnits.Append(",");
				BOL.BloodUnit bu = (BloodUnit) units[i];
				string guid = ((BloodUnit)units[i]).BloodUnitGuid.ToString();
				sbUnits.Append(guid);
			}

			// CR2899: changed input parameter
			// Gets all test results for all units.
			DataTable dtAllUnitTestResults = BOL.BloodUnitTest.GetBloodUnitTests(sbUnits.ToString(), ((BloodUnit)units[0]).DivisionCode);
			
			// Gets all antigens associated with the unit.  These are entered in UC01 (incoming shipment)
			// or UC70 (edit unit information).
			DataTable dtAllUnitAntigens = BOL.BloodUnitAntigen.GetBloodUnitAntigens(sbUnits.ToString(), ((BloodUnit)units[0]).DivisionCode);



			// Now, iterate through the units and get results
			for (int idx=0; idx<units.Count; idx++)
			{
				BOL.BloodUnit bloodUnit = new BOL.BloodUnit( ((BloodUnit) units[idx]).BloodUnitGuid);
				
				// Filter out the test results and antigens for the individual unit.
				DataRow [] dtRowUnitTestResults = dtAllUnitTestResults.Select(TABLE.BloodUnitTest.BloodUnitGuid + " = '" + bloodUnit.BloodUnitGuid.ToString() + "'");
				DataRow [] dtRowUnitAntigens = dtAllUnitAntigens.Select(TABLE.BloodUnitTest.BloodUnitGuid + " = '" + bloodUnit.BloodUnitGuid.ToString() + "'");


				// Convert DataRow array to DataTable for future selects
				DataTable dtUnitTestResults = dtAllUnitTestResults.Clone();
				foreach (DataRow dtRow in dtRowUnitTestResults)
				{
					dtUnitTestResults.ImportRow(dtRow);
				}
				dtUnitTestResults.AcceptChanges();


				// Convert DataRow array to DataTable for future selects
				DataTable dtUnitAntigens = dtAllUnitAntigens.Clone();
				foreach (DataRow dtRow in dtRowUnitAntigens)
				{
					dtUnitAntigens.ImportRow(dtRow);
				}
				dtUnitAntigens.AcceptChanges();




	
				// Now, compare patient antigens and antibodies to the unit's
				if ( !BOL.BloodUnit.IsAutologous(bloodUnit.DonationType.DonationTypeCode) &&
					( PatientAntigensCache.Length > 0 || 
					(ClinicallySignificantAntibodiesCache != null && ClinicallySignificantAntibodiesCache.Rows.Count > 0) ))
				{
					// First, compare the ANTIGENS
					foreach (DataRow drPatientAntigen in PatientAntigensCache)
					{					
						int antigenTypeId = (int) drPatientAntigen[TABLE.PatientTransfusionRequirement.AntigenTypeId];
						string antigenName = (string) drPatientAntigen[TABLE.PatientTransfusionRequirement.TransfusionRequirementText];

						// Ignore if D antigen and unit is Rh Neg
						if (antigenTypeId == (int) Common.Antigen.D && bloodUnit.BloodUnitMedia.AboRh.RH == Common.RH.Negative) continue;

						if (alreadyAdded.Contains(antigenTypeId)) continue;

						// _dtAntigenTests represents the AntigenTest table
						// 2) Get a list of tests associated with the AntigenTypeId
						DataRow [] drTests = AntigenTestsCache.Select(VbecsTables.BloodUnitAntigen.AntigenTypeId + " = " + antigenTypeId);

						// if there are tests available for this antigen, check these first
						if (drTests.Length > 0)
						{
							nCount = drTests.Length;
							for(int i=0; i<nCount; i++)
							{
								int bloodTestTypeID = (int) drTests[i][VbecsTables.BloodTestType.BloodTestTypeId];
					
								// Now, see if the unit has been tested for this antigen.  drUnitTestResults is derived from the BloodUnitTest table						
								DataRow [] drUnitTestResult = dtUnitTestResults.Select(VbecsTables.BloodTestType.BloodTestTypeId + " = " +  bloodTestTypeID);
						
								// No test results (BR_3.15)
								if (drUnitTestResult.Length != 0)
								{
									// Positive or Inconclusive test result (BR_3.06)
									// Condition #5 (unit testing)
									Common.TestResult testResult = Common.Utility.GetTestResultEnumFromString(drUnitTestResult[0][VbecsTables.BloodUnitTest.TestResultId].ToString());
									if ((testResult == Common.TestResult.P || testResult == Common.TestResult.I) && !alreadyAdded.Contains(antigenTypeId))
									{
										bloodUnitAntigens.Add(new BOL.AntigenNegativeMessage(antigenTypeId, antigenName, string.Empty, true, testResult));
										alreadyAdded.Add(antigenTypeId);
										break;
									}
									else if (testResult == Common.TestResult.N)
									{
										// Neg test result, so we're done
										// Condition #4 (unit testing)
										alreadyAdded.Add(antigenTypeId);
										break;
									}
								} //if
							}//for	
						}//if tests>0
					
						// if no test results were found, the BloodUnitAntigen table has to be checked (UC01, UC70)
						if (!alreadyAdded.Contains(antigenTypeId))
						{
							// Check the BloodUnitAntigen table also - This is from UC01 
							// Condition #3 (unit testing)
							DataRow [] drBloodUnitAntigenResult = dtUnitAntigens.Select(VbecsTables.BloodUnitAntigen.AntigenTypeId + " = " + antigenTypeId + " AND " + VbecsTables.BloodUnitAntigen.Positive + " = '" + true + "'" );
							if (drBloodUnitAntigenResult.Length > 0)
							{
								bloodUnitAntigens.Add(new BOL.AntigenNegativeMessage(antigenTypeId, antigenName, string.Empty, true, Common.TestResult.P));
								alreadyAdded.Add(antigenTypeId);
							}
							else
							{
								// Make sure unit has not tested negative for the antigen
								// Condition #2 (unit testing)
								drBloodUnitAntigenResult = dtUnitAntigens.Select(VbecsTables.BloodUnitAntigen.AntigenTypeId + " = " + antigenTypeId + " AND " +  VbecsTables.BloodUnitAntigen.Positive + " = '" + false + "'" );
								if (drBloodUnitAntigenResult.Length > 0)
								{
									alreadyAdded.Add(antigenTypeId);
								}
								// Condition #1 (unit testing)
								else
								{
									bloodUnitAntigens.Add(new BOL.AntigenNegativeMessage(antigenTypeId, antigenName, string.Empty, false, Common.TestResult.Unknown));
									alreadyAdded.Add(antigenTypeId);
								}
							}
						}
					}
					// Next, compare ANTIBODIES
					foreach (DataRow drPatientAntibody in ClinicallySignificantAntibodiesCache.Rows)
					{
						int antibodyTypeID = (int) drPatientAntibody[VbecsTables.PatientTransfusionRequirement.AntibodyTypeId]; 

						//BR_3.08
						//If the antibody ID is Anti-D and the unit is Rh Neg then we can skip the checks
						//because Rh Neg means that the unit is D Neg.
						if ((antibodyTypeID == (int)Common.Antibody.AntiD) && (bloodUnit.BloodUnitMedia.AboRh.RH == Common.RH.Negative))
							continue;
					
						DataRow [] drAntibodies = AntibodyTypesCache.Select(VbecsTables.PatientTransfusionRequirement.AntibodyTypeId + " = " + antibodyTypeID);
					
						// There may be multiple Antigen Negative requirements from one Antibody (TT_23.01).  Anti-f is an 
						// example (creates an antigen negative requirement for c and e).  This Antibody has 2 related antigen 
						// negative requirements, so both must be tested and be negative.
						for (int i = 0; i<drAntibodies.Length; i++)
						{
							string antigenName = drAntibodies[i][VbecsTables.AntigenType.AntigenTypeName].ToString();
							int antigenTypeID = (int) drAntibodies[i][VbecsTables.BloodUnitAntigen.AntigenTypeId];
							string antibodyName = (string) drAntibodies[i][VbecsTables.AntibodyType.AntibodyTypeName];
														
							if (alreadyAdded.Contains(antigenTypeID) || 
								(antigenTypeID == (int) Common.Antigen.D && bloodUnit.BloodUnitMedia.AboRh.RH == Common.RH.Negative) )   continue;

							// Get a list of tests associated with the AntigenTypeId
							DataRow [] drTests = AntigenTestsCache.Select(VbecsTables.BloodUnitAntigen.AntigenTypeId + " = " + antigenTypeID);

							// if there are tests for this antigen
							if (drTests.Length > 0)
							{
								for (int j=0; j<drTests.Length; j++)
								{
									int bloodTestTypeID = (int) drTests[j][VbecsTables.BloodTestType.BloodTestTypeId];
				
									// Now, see if the unit has been tested for this antigen.
									DataRow [] drUnitTestResult = dtUnitTestResults.Select(VbecsTables.BloodTestType.BloodTestTypeId + " = " +  bloodTestTypeID);
					
									// No test results (BR_3.15)
									if (drUnitTestResult.Length > 0)
									{
										// Positive or Inconclusive test result (BR_3.06)
										// Condition #5 (unit testing)
										Common.TestResult testResult = Common.Utility.GetTestResultEnumFromString(drUnitTestResult[0][VbecsTables.BloodUnitTest.TestResultId].ToString());
										if ((testResult == Common.TestResult.P || testResult == Common.TestResult.I) && !alreadyAdded.Contains(antigenTypeID))
										{										
											bloodUnitAntigens.Add(new BOL.AntigenNegativeMessage(antigenTypeID, antigenName, antibodyName, true, testResult)); //CR2997
											alreadyAdded.Add(antigenTypeID);
										
											break;
										}
											// Condition #4 (unit testing)
										else if (testResult == Common.TestResult.N)
										{
											alreadyAdded.Add(antigenTypeID);
											break;
										}
									} //if
								} //for
							}//if

							// Next, check the BloodUnitAntigen table (UC01)
							if(!alreadyAdded.Contains(antigenTypeID))
							{
								// Check the BloodUnitAntigen table also
								DataRow [] drBloodUnitAntigenResult = dtUnitAntigens.Select(VbecsTables.BloodUnitAntigen.AntigenTypeId + " = " + antigenTypeID + " AND " + VbecsTables.BloodUnitAntigen.Positive + " = '" + true + "'" );
								if (drBloodUnitAntigenResult.Length > 0) // positive test result
								{
									bloodUnitAntigens.Add(new BOL.AntigenNegativeMessage(antigenTypeID, antigenName, antibodyName, true, Common.TestResult.P));
									alreadyAdded.Add(antigenTypeID);
								}
								else
								{
									// Make sure unit has not tested negative for the antigen
									// Condition #3 (unit testing)
									drBloodUnitAntigenResult = dtUnitAntigens.Select(VbecsTables.BloodUnitAntigen.AntigenTypeId + " = " + antigenTypeID + " AND " +  VbecsTables.BloodUnitAntigen.Positive + " = '" + false + "'" );
									if (drBloodUnitAntigenResult.Length > 0)
									{
										alreadyAdded.Add(antigenTypeID);
									}
										// Condition #1 (unit testing)
									else
									{
										bloodUnitAntigens.Add(new BOL.AntigenNegativeMessage(antigenTypeID, antigenName, antibodyName, false, Common.TestResult.Unknown));
										alreadyAdded.Add(antigenTypeID);
									}
								}
							} // if
						} // for
					}
				}
				result.Add(bloodUnit.BloodUnitGuid, bloodUnitAntigens);
				bloodUnitAntigens = new ArrayList();
				alreadyAdded = new ArrayList();
			} // for
			
			return result;
		}

		// CR2899 end

		

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/13/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3866"> 
		///		<ExpectedInput>ComonentClass, BloodUnit</ExpectedInput>
		///		<ExpectedOutput>bool</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3883"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// ValidateProductType
		/// BR_3.33
		/// </summary>
		/// <param name="componentClass"></param>
		/// <param name="bu"></param>
		/// <returns></returns>
		public bool ValidateProductType(Common.ComponentClass componentClass, BOL.BloodUnit bu)
		{
			//Call to BR_3.33 this is tied to DR 4844 
			bool valid = true;
			switch(componentClass)
			{
				case Common.ComponentClass.WB:
				{
					//TT_3.04 portion that applies always regardless of TR's
					//E001-E009 and E049-E054
					string[] wbRbcProductTypes = new string[]{Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.WholeBlood),
																 Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.RedBloodCells),
																 Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.WashedRedBloodCells),
																 Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.FrozenRedBloodCells),
																 Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.FrozenRejuvenatedRedBloodCells),
																 Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.DeglycerolizedRedBloodCells),
																 Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.DeglycerolizedRejuvenatedRedBloodCells),
																 Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.RejuvenatedRedBloodCells),
																 Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.ApheresisRedBloodCells),
																 Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.WashedApheresisRedBloodCells),
																 Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.FrozenApheresisRedBloodCells),
																 Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.DeglycerolizedApheresisRedBloodCells),
																 Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.RejuvenatedApheresisRedBloodCells),
																 Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.FrozenRejuvenatedApheresisRedBloodCells),
																 Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.DeglycerolizedRejuvenatedApheresisRedBloodCells)};
					valid = this.ValidProductTypesForAssign(bu, wbRbcProductTypes, bu.ProductType.ProductTypeCode);
					break;
				}
				case Common.ComponentClass.RBC:
				{
					//TT_3.04 portion that applies always regardless of TR's
					//E001-E009 and E049-E054
					string[] wbRbcProductTypes = new string[]{Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.WholeBlood),
																 Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.RedBloodCells),
																 Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.WashedRedBloodCells),
																 Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.FrozenRedBloodCells),
																 Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.FrozenRejuvenatedRedBloodCells),
																 Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.DeglycerolizedRedBloodCells),
																 Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.DeglycerolizedRejuvenatedRedBloodCells),
																 Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.RejuvenatedRedBloodCells),
																 Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.ApheresisRedBloodCells),
																 Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.WashedApheresisRedBloodCells),
																 Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.FrozenApheresisRedBloodCells),
																 Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.DeglycerolizedApheresisRedBloodCells),
																 Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.RejuvenatedApheresisRedBloodCells),
																 Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.FrozenRejuvenatedApheresisRedBloodCells),
																 Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.DeglycerolizedRejuvenatedApheresisRedBloodCells)};
					valid = this.ValidProductTypesForAssign(bu, wbRbcProductTypes, bu.ProductType.ProductTypeCode);
					break;
				}
				case Common.ComponentClass.FFP:
				{
					//DR 4844 formerlyTT_3.04 portion that applies always regardless of TR's
					//E010-E013
					string[] ffpProductTypes = new string[]{Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.FreshFrozenPlasma),
															   Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.ThawedFreshFrozenPlasma),
															   Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.ApheresisFreshFrozenPlasma),
															   Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.ThawedApheresisFreshFrozenPlasma)};
																
					valid = this.ValidProductTypesForAssign(bu, ffpProductTypes, bu.ProductType.ProductTypeCode);
					break;
				}
				case Common.ComponentClass.CRYO:
				{
                    //DR 4844 formerly TT_3.04 portion that applies always regardless of TR's
					//E028-E033
					string[] cryoProductTypes = new string[]{Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.Cryoprecipitate),
																Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.ThawedCryoprecipitate),
																Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.PooledCryoprecipitate),
																Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.ThawedPooledCryoprecipitate),
																Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.ApheresisCryoprecipitate),
																Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.ThawedApheresisCryoprecipitate)};
					valid = this.ValidProductTypesForAssign(bu, cryoProductTypes, bu.ProductType.ProductTypeCode);
					break;
				}
				case Common.ComponentClass.Other:
				{
                    //DR 4844 formerly TT_3.04 portion that applies always regardless of TR's
					//E014-E015,E017-E019,E034-E045
					string[] otherProductTypes = new string[]{Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.ApheresisPlasma),
																 Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.ThawedApheresisPlasma),
																 Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.Plasma),
																 Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.ThawedPlasma),
																 Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.PlateletRichPlasma),
																 Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.Granulocytes),
																 Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.ApheresisGranulocytes),
																 Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.PooledGranulocytes),
																 Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.ApheresisGranulocytesPlatelets),
																 Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.Leukocytes),
																 Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.ApheresisLeukocytes),
																 Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.PooledPlasma),
																 Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.LiquidApheresisPlasma),
																 Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.PlateletRichBuffyCoat),
																 Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.PooledPlateletRichBuffyCoat),
																 Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.ApheresisLymphocytes),
																 Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.ApheresisMonocytes)};
					valid = this.ValidProductTypesForAssign(bu, otherProductTypes, bu.ProductType.ProductTypeCode);
					break;
				}
				case Common.ComponentClass.PLT:
				{
                    //DR 4844 formerly TT_3.04 portion that applies always regardless of TR's
					//E020-E027
					string[] pltProductTypes = new string[]{Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.Platelets),
															   Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.WashedPlatelets),
															   Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.PooledPlatelets),
															   Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.WashedPooledPlatelets),
															   Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.ApheresisPlatelets),
															   Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.FrozenApheresisPlatelets),
															   Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.ThawedApheresisPlatelets),
															   Common.Utility.GetProductTypeCodeFromProductTypeEnum(Common.ProductType.WashedApheresisPlatelets)};
					valid = this.ValidProductTypesForAssign(bu, pltProductTypes, bu.ProductType.ProductTypeCode);
					break;
				}
			}
			return valid;
		}

        ///<Developers>
        ///	<Developer>C. Jensen</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>11/14/2012</CreationDate>
        ///<TestCases>
        ///	
        ///<Case type="0" testid ="7158"> 
        ///		<ExpectedInput>BloodUnit</ExpectedInput>
        ///		<ExpectedOutput>true</ExpectedOutput>
        ///	</Case>
        ///
        ///<Case type="1" testid ="7159"> 
        ///		<ExpectedInput>BloodUnit</ExpectedInput>
        ///		<ExpectedOutput>false</ExpectedOutput>
        ///	</Case>
        ///
        ///</TestCases>
        ///<Update></Update>
        ///<ArchivePlan></ArchivePlan>
        ///<Interfaces></Interfaces>
        /// <summary>
        /// IsUnitRhSatisfactory
        /// BR_3.21, BR_73.16
        /// </summary>
        /// <param name="bu"></param>
        /// <returns>bool</returns>
        // CR3261
        public bool IsUnitRhSatisfactory(BOL.BloodUnit bu)
        {
            if (((Common.ComponentClass)bu.BloodComponentClassId == Common.ComponentClass.RBC ||
                (Common.ComponentClass)bu.BloodComponentClassId == Common.ComponentClass.WB ||
                (Common.ComponentClass)bu.BloodComponentClassId == Common.ComponentClass.PLT ||
                (Common.ComponentClass)bu.BloodComponentClassId == Common.ComponentClass.Other)
                && ((bu.BloodUnitMedia.AboRh.RH == Common.RH.Positive || bu.BloodUnitMedia.AboRh.RH == Common.RH.PooledRH)
                && this.AboRh.RH != Common.RH.Positive))
            {
                return false;
            }
            else
            {
                return true;
            }
        }
		/// <summary>
		/// ValidProductTypesForAssign
		/// </summary>
		/// <param name="bu"></param>
		/// <param name="productTypeValidList"></param>
		/// <param name="productTypeCode"></param>
		/// <returns></returns>
		private bool ValidProductTypesForAssign(BOL.BloodUnit bu, string[] productTypeValidList, string productTypeCode)
		{
			foreach(string validProductType in productTypeValidList) 
			{
				if(productTypeCode == validProductType)
				{
					return true;
				}
			}
			return false;
		}



		// CR2899 start

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/13/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3189"> 
		///		<ExpectedInput>Valid bloodunitguid</ExpectedInput>
		///		<ExpectedOutput>Arraylist</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3190"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Checks for unfulfilled component requirements BR_56.25
		/// </summary>
		/// <param name="bloodUnitGuid"></param>
		/// <returns></returns>
		public ArrayList ValidateComponentRequirements(Guid bloodUnitGuid)
		{
			ArrayList units = new ArrayList(1);
			units.Add(new BOL.BloodUnit(bloodUnitGuid));

			return (ArrayList) (ValidateComponentRequirements(units))[bloodUnitGuid];
		}



		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/13/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5917"> 
		///		<ExpectedInput>Valid bloodunitguid</ExpectedInput>
		///		<ExpectedOutput>Arraylist</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5918"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Checks for unfulfilled component requirements BR_56.25
		/// </summary>
		/// <param name="units">Datatable of units</param>
		/// <returns>Hashtable</returns>
		public Hashtable ValidateComponentRequirements(ArrayList units)
		{
			Hashtable crHash = new Hashtable();
			ArrayList cRsNotMet = new ArrayList();
			// This is TT_3.04 portion that deals with the TR's
			System.Data.DataTable dtTRs = this.TransfusionRequirements;

			// Get active Component requirements for patient
			System.Data.DataRow [] dtRows = dtTRs.Select(VbecsTables.PatientTransfusionRequirement.TransfusionRequirementCategoryCode + " = 'CR' AND RecordStatusCode = 'A'");
		
			int compReqCount = dtRows.GetLength(0);

			// no requirements, so stop here
			if (compReqCount == 0) return crHash;


			for(int i=0; i<units.Count; i++)
			{			
				BOL.BloodUnit bloodUnit = (BOL.BloodUnit) units[i];
				for(int nCtr = 0; nCtr < compReqCount; nCtr++)
				{
					System.Data.DataRow dtRow = dtRows[nCtr];
					switch((byte) dtRow[VbecsTables.ComponentRequirement.ComponentRequirementId])
					{
						case (int) Common.ComponentRequirement.IRRADIATE:
						{
							if (bloodUnit.IsComponentRequirementSatisfied(Common.ComponentRequirement.IRRADIATE) == false)
							{
								cRsNotMet.Add("Irradiated");
							}
							break;
						}
						case (int) Common.ComponentRequirement.WASHEDRBC:
						{
							if (bloodUnit.IsComponentRequirementSatisfied(Common.ComponentRequirement.WASHEDRBC) == false)
							{
								cRsNotMet.Add("Washed RBC");
							}
							break;
						}
						case (int) Common.ComponentRequirement.WASHEDPLT:
						{
							if (bloodUnit.IsComponentRequirementSatisfied(Common.ComponentRequirement.WASHEDPLT) == false)
							{
								cRsNotMet.Add("Washed Platelet");
							}
							break;
						}
						case (int) Common.ComponentRequirement.LEUKORED:
						{
							if (bloodUnit.IsComponentRequirementSatisfied(Common.ComponentRequirement.LEUKORED) == false)
							{
								cRsNotMet.Add("Leukoreduced");
							}
							break;
						}
						case (int) Common.ComponentRequirement.SCNEG:
						{
							if (bloodUnit.IsComponentRequirementSatisfied(Common.ComponentRequirement.SCNEG) == false)
							{
								cRsNotMet.Add("SC Negative");
							}
							break;
						}
						case (int) Common.ComponentRequirement.CMVNEG:
						{
							if (bloodUnit.IsComponentRequirementSatisfied(Common.ComponentRequirement.CMVNEG) == false)
							{
								cRsNotMet.Add("CMV Negative");
							}
							break;
						}
					}
				} // for
				crHash.Add(bloodUnit.BloodUnitGuid, cRsNotMet);
				// new instance for next unit
				cRsNotMet = new ArrayList();
			}
			return crHash;
		}

		

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/13/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3884"> 
		///		<ExpectedInput>Valid bloodunitguid</ExpectedInput>
		///		<ExpectedOutput>Arraylist</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3885"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Checks if patient has any positive, inconclusive or untested antibody or antigen negative requirements - BR_56.22
		/// </summary>
		/// <param name="bloodUnitGuid"></param>
		/// <returns></returns>
		public Common.TestResult CheckForPositiveAntibodyAntigenRequirements(Guid bloodUnitGuid)
		{
			ArrayList AntigenAndAntibodyReq = this.ValidateAntibodyAntigenRequirementsMet(new BOL.BloodUnit(bloodUnitGuid));

			foreach (BOL.AntigenNegativeMessage antigen in AntigenAndAntibodyReq)
			{
				if (antigen.Tested == false)
				{
					return Common.TestResult.Unknown;
				}
				else
				{
					if (antigen.TestResult == Common.TestResult.P || antigen.TestResult == Common.TestResult.I)
					{
						return antigen.TestResult;
					}
				}
			}

			return Common.TestResult.N;
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/24/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4007"> 
		///		<ExpectedInput>Valid patient object with unique identifier set</ExpectedInput>
		///		<ExpectedOutput>Array list of specimen information</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4008"> 
		///		<ExpectedInput>Patient object without unique identifier set</ExpectedInput>
		///		<ExpectedOutput>Argument Exception thrown</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets a complete list of specimens associated with the patient
		/// Implements BR_48.02
		/// </summary>
		/// <param name="startDate">Start date</param>
		/// <param name="endDate">End date</param>
		/// <returns>Array list of specimens drawn on patient</returns>
		public System.Collections.ArrayList GetAllSpecimensForPatient(DateTime startDate, DateTime endDate)
		{
			if (this.PatientGuid == Guid.Empty)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("patient unique identifier").ResString);
			}
			//
			DataTable dt = DAL.Patient.GetAllSpecimensForPatient(this.PatientGuid, startDate.Date, endDate.Date.AddHours(23).AddMinutes(59).AddSeconds(59));
			//
			System.Collections.ArrayList specimens = new System.Collections.ArrayList(dt.Rows.Count);
			//
			foreach(DataRow dr in dt.Rows)
			{
				BOL.Specimen specimen = new BOL.Specimen(dr);
				specimens.Add(specimen);
			}
			//
			return specimens;
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/24/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3193"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>int</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3886"> 
		///		<ExpectedInput>Patient object without unique identifier set</ExpectedInput>
		///		<ExpectedOutput>Argument Exception thrown</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Returns a count of all specimens associated with the given patient
		/// </summary>
		/// <returns>Specimen count</returns>
		public int GetAllSpecimensForPatientCount()
		{
			if (this.PatientGuid == Guid.Empty)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("patient unique identifier").ResString);
			}
			//
			return DAL.Patient.GetAllSpecimensForPatientCount(this.PatientGuid);
		}

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/22/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4123"> 
		///		<ExpectedInput>Patient with inconsistent ABO/Rh results</ExpectedInput>
		///		<ExpectedOutput>ArrayLists with length of 2 or 4</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4124"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns ArrayList of specimen tests as follows:
		/// 1)  If there is no inconsistency, an empty ArrayList is returned.
		/// 2)  If the most recent specimen results are inconsistent with the patient record,
		///		an ArrayList with 2 SpecimenTest objects is returned (one for ABO and another 
		///		for Rh).  In this case, the specimen results are inconsistent with the Patient 
		///		record (from conversion).
		///	3)  If the most recent specimen results are inconsistent with the previous specimen
		///		results, an ArrayList of 4 SpecimenTests is returned (the most recent ABO and 
		///		Rh results are the first objects in the ArrayList respectively).
		///		BR_100.01
		/// </summary>
		/// <returns></returns>
		public ArrayList GetPatientAboRHHistoryForJustification()
		{
			_aboHistory = new ArrayList();
			DataTable dt =  DAL.Patient.GetRecentPatientAboRHHistory(_patientGUID);
			DataRow [] drABOs = dt.Select(VbecsTables.BloodTestType.BloodTestTypeId + " = " + (int) Common.TestType.ABORepeat + " OR BloodTestTypeId = " + (int) Common.TestType.ABOInterpTAS + " OR BloodTestTypeId = " + (int) Common.TestType.ABOInterp, "TestDate DESC");
			DataRow [] drRhs = dt.Select(VbecsTables.BloodTestType.BloodTestTypeId + " = " + (int) Common.TestType.RhRepeat + " OR BloodTestTypeId = " + (int) Common.TestType.RhInterpTAS  + " OR BloodTestTypeId = " + (int) Common.TestType.RhInterp, "TestDate DESC");

			// if 0 results, there is no inconsistency and therefore, no justification
			if (dt.Rows.Count == 0)
			{
				return _aboHistory;
			}
				// When row count equals 2, compare specimen test with patient record
			else if(dt.Rows.Count==2 && drABOs.Length==1 && drRhs.Length==1 && _conversionABORh != null)
			{
				Common.ABO testABO = Common.Utility.GetAboFromString( drABOs[0][VbecsTables.BloodUnitTest.TestResultId].ToString().Trim() );
				Common.RH testRh = Common.Utility.GetRhFromString( drRhs[0][VbecsTables.BloodUnitTest.TestResultId].ToString().Trim() );
				
				// If it's already been justified, return an empty ArrayList
				if ( (bool) drABOs[0][VbecsTables.SpecimenTest.AboRhChangeIndicator] == true) return _aboHistory; 

				if ( (_conversionABORh.Abo != Common.ABO.NA && 
					(_conversionABORh.RH != Common.RH.Blank || _conversionABORh.RH != Common.RH.NotProvided)) &&
					(_conversionABORh.Abo != testABO || this._conversionABORh.RH != testRh))
				{
					_aboHistory.Add(new SpecimenTest(drABOs[0]));
					_aboHistory.Add(new SpecimenTest(drRhs[0]));
				}
			}
				// When row count equals 4, compare specimen tests 
			else if (dt.Rows.Count==4 && drABOs.Length==2 && drRhs.Length==2)
			{
				Common.ABO currentABO = Common.Utility.GetAboFromString( drABOs[0][VbecsTables.BloodUnitTest.TestResultId].ToString().Trim() );
				Common.RH currentRh = Common.Utility.GetRhFromString( drRhs[0][VbecsTables.BloodUnitTest.TestResultId].ToString().Trim() );
				
				// If it's already been justified, return an empty ArrayList
				if ( (bool) drABOs[0][VbecsTables.SpecimenTest.AboRhChangeIndicator] == true)
				{
					return _aboHistory;
				}

				//If the results are from the same specimen, return an empty ArrayList (BR_100.07)
				if ((Guid)drABOs[0][VbecsTables.SpecimenTest.PatientSpecimenGuid] == (Guid)drABOs[1][VbecsTables.SpecimenTest.PatientSpecimenGuid])
				{
					return _aboHistory;
				}

				if ( currentABO != Common.Utility.GetAboFromString( drABOs[1][VbecsTables.BloodUnitTest.TestResultId].ToString().Trim()) ||
					currentRh != Common.Utility.GetRhFromString( drRhs[1][VbecsTables.BloodUnitTest.TestResultId].ToString().Trim()) )
				{
					_aboHistory.Add(new SpecimenTest(drABOs[0]));
					_aboHistory.Add(new SpecimenTest(drRhs[0]));
					_aboHistory.Add(new SpecimenTest(drABOs[1]));
					_aboHistory.Add(new SpecimenTest(drRhs[1]));
				}
			}
			return _aboHistory;
		}

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/22/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4121"> 
		///		<ExpectedInput>Valid arguements</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4122"> 
		///		<ExpectedInput>Invalid arguments</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// JustifyPatientABORhChange
		/// BR_100.03
		/// </summary>
		/// <param name="abo"></param>
		/// <param name="rh"></param>
		/// <param name="justificationReason"></param>
		/// <returns></returns>
		public bool JustifyPatientABORhChange(string abo, string rh, string justificationReason)
		{
			if (abo.Trim() == string.Empty || rh.Trim() == string.Empty || justificationReason.Trim() == string.Empty)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("ABO, Rh or Justification Reason").ResString);
			}

			// Create DataTables for save
			DataTable dtPatient = new DataTable();
			dtPatient.Columns.Add(TABLE.Patient.PatientGuid, typeof(System.Guid));
			dtPatient.Columns.Add(TABLE.Patient.AboRhChangeIndicator, typeof(byte));
			dtPatient.Columns.Add(TABLE.Patient.LastUpdateUser, typeof(string));
			dtPatient.Columns.Add(TABLE.Patient.RowVersion, typeof(byte[]));
			
			DataTable dtAboSpecimenTest = new DataTable();
			dtAboSpecimenTest.Columns.Add(TABLE.SpecimenTest.SpecimenTestGuid, typeof(System.Guid));
			dtAboSpecimenTest.Columns.Add(TABLE.SpecimenTest.TestResultId, typeof(string));
			dtAboSpecimenTest.Columns.Add(TABLE.SpecimenTest.AboRhChangeIndicator, typeof(int));
			dtAboSpecimenTest.Columns.Add(TABLE.SpecimenTest.AboRhChangeJustification, typeof(string));
			dtAboSpecimenTest.Columns.Add(TABLE.SpecimenTest.LastUpdateUser, typeof(string));
			dtAboSpecimenTest.Columns.Add(TABLE.SpecimenTest.RowVersion, typeof(byte[]));

			DataTable dtRhSpecimenTest = new DataTable();
			dtRhSpecimenTest.Columns.Add(TABLE.SpecimenTest.SpecimenTestGuid, typeof(System.Guid));
			dtRhSpecimenTest.Columns.Add(TABLE.SpecimenTest.TestResultId, typeof(string));
			dtRhSpecimenTest.Columns.Add(TABLE.SpecimenTest.AboRhChangeIndicator , typeof(int));
			dtRhSpecimenTest.Columns.Add(TABLE.SpecimenTest.AboRhChangeJustification, typeof(string));
			dtRhSpecimenTest.Columns.Add(TABLE.SpecimenTest.LastUpdateUser, typeof(string));
			dtRhSpecimenTest.Columns.Add(TABLE.SpecimenTest.RowVersion, typeof(byte[]));

			
			// Populate DataTables
			DataRow drPatient = dtPatient.NewRow();
			drPatient[TABLE.Patient.PatientGuid] = this.PatientGuid;
			drPatient[TABLE.Patient.AboRhChangeIndicator] = 1;
			drPatient[TABLE.Patient.LastUpdateUser] = Common.LogonUser.LogonUserName;
			drPatient[TABLE.Patient.RowVersion] = this.RowVersion;
			dtPatient.Rows.Add(drPatient);

			SpecimenTest aboSpecimenTest = (SpecimenTest) _aboHistory[0];
			SpecimenTest rhSpecimenTest = (SpecimenTest) _aboHistory[1];
				 
			DataRow drAboSpecimenTest = dtAboSpecimenTest.NewRow();
			drAboSpecimenTest[TABLE.SpecimenTest.SpecimenTestGuid] = aboSpecimenTest.SpecimenTestGuid;
			drAboSpecimenTest[TABLE.SpecimenTest.TestResultId] = abo;
			drAboSpecimenTest[TABLE.SpecimenTest.AboRhChangeIndicator] = 1;
			drAboSpecimenTest[TABLE.SpecimenTest.AboRhChangeJustification] = justificationReason;
			drAboSpecimenTest[TABLE.SpecimenTest.LastUpdateUser] = Common.LogonUser.LogonUserName;
			drAboSpecimenTest[TABLE.SpecimenTest.RowVersion] = aboSpecimenTest.RowVersion;
			dtAboSpecimenTest.Rows.Add(drAboSpecimenTest);

			DataRow drRhSpecimenTest = dtRhSpecimenTest.NewRow();
			drRhSpecimenTest[TABLE.SpecimenTest.SpecimenTestGuid] = rhSpecimenTest.SpecimenTestGuid;
			drRhSpecimenTest[TABLE.SpecimenTest.TestResultId] = rh;
			drRhSpecimenTest[TABLE.SpecimenTest.AboRhChangeIndicator] = 1;
			drRhSpecimenTest[TABLE.SpecimenTest.AboRhChangeJustification] = justificationReason;
			drRhSpecimenTest[TABLE.SpecimenTest.LastUpdateUser] = Common.LogonUser.LogonUserName;
			drRhSpecimenTest[TABLE.SpecimenTest.RowVersion] = rhSpecimenTest.RowVersion;
			dtRhSpecimenTest.Rows.Add(drRhSpecimenTest);
			System.Guid [] patientGuids = new Guid[1];
			patientGuids[0] = this.PatientGuid;
			//CR 2212
			System.Data.DataTable dtWorkloadEvents = BOL.WorkloadEvent.GenerateWorkloadData(Common.WorkloadProcessID.Justification, patientGuids, this.PatientGuid, Common.WorkloadTransactionType.Miscellanious, false);
		
			// Save!
			return (DAL.Patient.JustifyAboRHChange(dtPatient, dtAboSpecimenTest, dtRhSpecimenTest, Common.UpdateFunction.UC100FrmJustifyAboRHChange, dtWorkloadEvents));
		}

		/// <summary>
		/// This returns the compatibility string.  Compatibility is calculated by getting the compatibility
		/// percentages for the patient blood type and antigen negative requirements (system generated too).
		/// Unfortunately, compatibility percentage is incorrectly associated with antibodies in the database.
		/// As a result, the code is convoluted.
		/// </summary>
		/// <returns></returns>
		private string GetCompatibilityPercentage()
		{  
			DataTable dtAntibodies = BOL.AntibodyType.GetAntibodyTypeLookupList();
			int nCount = 0;
			this._dtTransfusionRequirements =  this.TransfusionRequirements;
			decimal compatibility = GetCompatibilityForBloodType();
			ArrayList added = new ArrayList();
			 
			if (compatibility == Decimal.MinValue) 
			{
				return "NA";
			}
			else
			{
				nCount = this._dtTransfusionRequirements.Rows.Count;
				for(int i=0; i<nCount; i++)
				{
					DataRow dtRow = this.TransfusionRequirements.Rows[i];
					Common.TransfusionRequirementCategoryCode catCode = Common.Utility.GetTransfusionCategoryCodeFromString(dtRow[TABLE.PatientTransfusionRequirement.TransfusionRequirementCategoryCode].ToString());
					Common.RecordStatusCode status = Common.Utility.GetRecordStatusCodeFromString(dtRow[TABLE.PatientTransfusionRequirement.RecordStatusCode].ToString());
					//CR 2998 Added check for converted antibodies to include them in calculation
					if (catCode == Common.TransfusionRequirementCategoryCode.ComponentRequirement || 
						(status != Common.RecordStatusCode.Active && status != Common.RecordStatusCode.Converted)) continue;

					// Process antibodies first
					if (catCode == Common.TransfusionRequirementCategoryCode.AntibodyIdentified)
					{
						int antibodyTypeId = dtRow.IsNull(TABLE.PatientTransfusionRequirement.AntibodyTypeId) ? 0 : (int) dtRow[TABLE.PatientTransfusionRequirement.AntibodyTypeId];
						
						// Get corresponding antigen neg IDs:
						DataRow [] dtRowAntibodies = dtAntibodies.Select(TABLE.PatientTransfusionRequirement.AntibodyTypeId + " = " + antibodyTypeId);
												
						for (int j=0; j<dtRowAntibodies.Length; j++)
						{
							int antigenTypeId = (int) dtRowAntibodies[j][TABLE.AntigenType.AntigenTypeId];
							
							// don't add again if already identified
							if (!added.Contains(antigenTypeId))
							{
								// Now, we must filter again to get the correct Compatibility Percentage.  Some antigens 
								// appear more than once in this table because they are associated with more than one 
								// antibody.  We must get the Compatibility Percentage that is a non null-value.
								DataRow [] dtRowAntigens = dtAntibodies.Select(TABLE.AntigenTest.AntigenTypeId + " = '" + antigenTypeId + "' AND " + TABLE.AntibodyParameter.CompatibilityPercentage + " IS NOT NULL");
						
								decimal compatibilityPercentage = Convert.ToDecimal( (byte) dtRowAntigens[0][TABLE.AntibodyParameter.CompatibilityPercentage])/100;
							
								if (compatibilityPercentage == 0) return "0";
								else 
								{
									compatibility = compatibility*compatibilityPercentage;
									added.Add(antigenTypeId);
								}
							}
						}
					}
						// Process antigens
					else
					{
						int antigenTypeId = dtRow.IsNull(TABLE.PatientTransfusionRequirement.AntigenTypeId) ? 0 : (int) dtRow[TABLE.PatientTransfusionRequirement.AntigenTypeId];
						decimal compatibilityPercentage = Convert.ToDecimal(dtRow[TABLE.AntibodyParameter.CompatibilityPercentage])/100; 
						if (compatibilityPercentage == 0) return "0";
						else if (!added.Contains(antigenTypeId))
						{
							compatibility = compatibility*compatibilityPercentage;
							added.Add(antigenTypeId);
						}
					}
				}
				// round and convert to a percentage
				compatibility = Decimal.Round(compatibility, 2)*100;
				if (compatibility.ToString().IndexOf(".") > -1)
				{
					return compatibility.ToString().Substring(0, compatibility.ToString().IndexOf(".")); 
				}
				else
				{
					return compatibility.ToString();
				}
			}
		}

		/// <summary>
		/// TT_23.01B
		/// </summary>
		/// <returns></returns>
		private Decimal GetCompatibilityForBloodType()
		{
			Decimal compatibility = Decimal.MinValue;

			switch (_aboRH.Abo)
			{
				case Common.ABO.A:
					compatibility = .45M;
					break;
				case Common.ABO.B:
					compatibility = .15M;
					break;
				case Common.ABO.AB:
					compatibility = .06M;
					break;
				case Common.ABO.O:
					compatibility = .45M;
					break;
			}
			return compatibility;
		}

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/24/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8089"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>non-empty string</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="8090"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>empty string</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// GetPatientAboRhJustificationReason
		/// </summary>
		/// <returns></returns>
		public string GetPatientAboRhJustificationReason()
		{
			DataRow dtRow = DAL.Patient.GetPatientAboRhJustificationReason(_patientGUID);
			if (dtRow != null && !dtRow.IsNull(TABLE.SpecimenTest.AboRhChangeJustification))
			{
				return (string) DAL.Patient.GetPatientAboRhJustificationReason(_patientGUID)[TABLE.SpecimenTest.AboRhChangeJustification];
			}
			else
			{
				return string.Empty;
			}
		}
		

		#endregion

		#region Static Methods

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/11/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3196"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>ArrayList</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3903"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Get the list of treating specialties for the division
		/// </summary>
		/// <returns></returns>
		public static ArrayList GetTreatingSpecialtiesList()
		{
			DataTable dtTreatingSpecialties = DAL.Patient.GetTreatingSpecialtyList();
			ArrayList treatingSpecialtyList = new ArrayList();
			
			foreach(DataRow drTreatingSpecialties in dtTreatingSpecialties.Rows)
			{
				treatingSpecialtyList.Add(drTreatingSpecialties[VbecsTables.TreatingSpecialty.TreatingSpecialtyName].ToString());
			}

			return treatingSpecialtyList;
		}


		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/11/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="597"> 
		///		<ExpectedInput>Valid first name string (0-30 chars), middle initial (0-1 char) and last name (0-30 chars) strings</ExpectedInput>
		///		<ExpectedOutput>Patient name as single display string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1991"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Public function to return a name formatted for display from patient's last name, first name and middle name
		/// </summary>
		/// <param name="firstName">First Name</param>
		/// <param name="middleName">Middle Name</param>
		/// <param name="lastName">Last Name</param>
		/// <returns>Name as: LastName, FirstName MiddleName</returns>
		public static string BuildDisplayName(string firstName, string middleName, string lastName)
		{
			if (firstName == null)
			{
				firstName = string.Empty;
			}
			if (lastName == null)
			{
				lastName = string.Empty;
			}
			if (middleName == null)
			{
				middleName = string.Empty;
			}
			//
			string displayName = lastName.ToString();
			//
			//CR 2652 - removed code adding period (".") at the end of the name if the middle name was 1 character long
			//			replaced code using "+" to add strings with calls to string.concat method
			if (displayName.Length > 0 && ((firstName.ToString().Length > 0) || (middleName.ToString().Length > 0)))
			{
				displayName = string.Concat(displayName, ", ");
			}
			displayName = string.Concat(displayName, firstName.ToString());
			//
			if (middleName.ToString().Length > 0)
			{
				displayName = string.Concat(displayName, " ", middleName);
				//
			}
			return displayName;
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/11/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="598"> 
		///		<ExpectedInput>Valid patient name, SSN and accession number strings</ExpectedInput>
		///		<ExpectedOutput>Data table of matching patient entries</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1992"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Static function to call data access layer to search database for a patient,
		/// using (partial) last name, (partial) _ssn, or accession number.
		/// Implements BR_3.20
		/// </summary>
		/// <param name="patientName">Patient Name</param>
		/// <param name="patientSsn">Patient SSN</param>
		/// <param name="specimenUid">Specimen UID</param>
		/// <param name="divisionCode">divisionCode</param>
		/// <returns>Data table of patients matching search criteria</returns>
		public static DataTable Search(string patientName, string patientSsn, string specimenUid, string divisionCode)
		{
			return (DAL.Patient.Search(patientName,patientSsn,specimenUid,divisionCode));
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/10/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3526"> 
		///		<ExpectedInput>DataRow, UpdateFunction</ExpectedInput>
		///		<ExpectedOutput>bool</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3527"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///<summary>
		///InsertPatient
		///</summary>
		///<param name="drInputData">DatRow</param>
		///<param name="lastUpdateFunctionId">UC or calling method</param>
		public static bool InsertPatient(System.Data.DataRow drInputData, Common.UpdateFunction lastUpdateFunctionId)
		{
			string patientIcn = string.Empty;
			long vistaPatientId = -1;				//CR 2431	changed to long to allow for the database change of the VistaPatientId to Bigint
			bool patientIcnExistsInDb = false;
			bool vistaPatientIdExistsInDb = false;

			//Find out if they passed in a PatientIcn value
			if (drInputData.Table.Columns.Contains(TABLE.Patient.PatientIcn))
			{
				if (!drInputData.IsNull(TABLE.Patient.PatientIcn))
					patientIcn = drInputData[TABLE.Patient.PatientIcn].ToString();
			}

			//Find out if they passed in a VistaPatientId value
			if (drInputData.Table.Columns.Contains(TABLE.Patient.VistaPatientId))
			{
				if (!drInputData.IsNull(TABLE.Patient.VistaPatientId))
					//CR 2431	changed to Int64 to allow for the database change of the VistaPatientId to Bigint
					vistaPatientId =  System.Convert.ToInt64(drInputData[TABLE.Patient.VistaPatientId]);
			}

			//If out if the patientIcn exists already
			if (patientIcn.Trim() != string.Empty)
			{
				System.Data.DataTable dtIcnExists = DAL.Patient.GetPatientDetails(vistaPatientId, patientIcn);
				patientIcnExistsInDb = (dtIcnExists.Rows.Count > 0);
			}

			//Find out if the VistaPatientId exists already
			if (vistaPatientId != -1)
			{
				System.Data.DataTable dtPatientIdExists = DAL.Patient.GetPatientDetails(vistaPatientId, string.Empty);
				vistaPatientIdExistsInDb = (dtPatientIdExists.Rows.Count > 0);
			}

			if (!vistaPatientIdExistsInDb && !patientIcnExistsInDb)
			{
				System.Collections.ArrayList sprocArray = new ArrayList();
				System.Collections.ArrayList dtArray = new ArrayList();

				System.Data.DataTable dtPatient = DAL.Patient.GetEmptyPatientTableSchema(false);
				System.Data.DataRow drNewPatient = dtPatient.NewRow();

				//Build our datarow for spInsertPatient
				foreach(System.Data.DataColumn dc in dtPatient.Columns)
				{
					if (drInputData.Table.Columns.Contains(dc.ColumnName))
					{
						drNewPatient[dc.ColumnName] = drInputData[dc.ColumnName];
					}
				}
				
				drNewPatient[TABLE.Patient.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Active);
				dtPatient.Rows.Add(drNewPatient);
				dtArray.Add(Common.Utility.AppendLastUpdateInformation(dtPatient, lastUpdateFunctionId));

				//Get the PatientTreatment data -- If it's available
				System.Data.DataTable dtPatientTreatment = DAL.Patient.GetEmptyPatientTreatmentTableSchema(false);
				System.Data.DataRow drNewPatientTreatment = dtPatientTreatment.NewRow();

				//Build our datarow for spInsertPatientTreatment
				foreach(System.Data.DataColumn dc in dtPatientTreatment.Columns)
				{
					if (drInputData.Table.Columns.Contains(dc.ColumnName))
					{
						drNewPatientTreatment[dc.ColumnName] = drInputData[dc.ColumnName];
					}
				}

				drNewPatientTreatment[TABLE.PatientTreatment.PatientTreatmentGuid] = System.Guid.NewGuid();
				dtPatientTreatment.Rows.Add(drNewPatientTreatment);
				dtArray.Add(Common.Utility.AppendLastUpdateInformation(dtPatientTreatment, lastUpdateFunctionId));

				sprocArray.Add(Common.VbecsStoredProcs.InsertPatient.StoredProcName);
				sprocArray.Add(Common.VbecsStoredProcs.InsertPatientTreatment.StoredProcName);

				return (new Common.StoredProcedure().TransactionalGetValue(sprocArray, dtArray) == 0);
			}
			else
				return(true);		//The Patient already exists, so return TRUE
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/10/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3528"> 
		///		<ExpectedInput>patient name, ssn, uid</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3529"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// SearchVistALink
		/// </summary>
		/// <param name="patientName"></param>
		/// <param name="patientSsn"></param>
		/// <param name="specimenUid"></param>
		/// <returns></returns>
		public static DataTable SearchVistALink(string patientName, string patientSsn, string specimenUid)
		{			
			DataTable _dt = DAL.VAL.PatientData.GetVistaPatientData( patientName, patientSsn, specimenUid );

			_dt.Columns.Add( VbecsTables.Patient.PatientGuid, typeof( Guid ) );

			foreach(System.Data.DataRow dtRow in _dt.Rows)
				dtRow[VbecsTables.Patient.PatientGuid] = System.Guid.NewGuid();

			return _dt;
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/10/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3904"> 
		///		<ExpectedInput>datatables</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3201"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Saves special instruction and transfusion requirements
		/// BR_38.15
		/// </summary>
		public static bool SaveSIsAndTRs(DataTable insertSpecialInstructions, 
			DataTable updateSpecialInstructions,
			DataTable insertTransfusionRequirements, 
			DataTable updateTransfusionRequirements,
			Common.UpdateFunction lastUpdateFunctionId)
		{
			if (insertSpecialInstructions == null ||
				updateSpecialInstructions == null ||
				insertTransfusionRequirements == null ||
				updateTransfusionRequirements == null ||
				lastUpdateFunctionId == Common.UpdateFunction.Invalid)
			{
				throw new ArgumentException();
			}

			System.Collections.ArrayList _dtArrayList = new System.Collections.ArrayList();
			System.Collections.ArrayList _sprocArrayList = new System.Collections.ArrayList();


			_dtArrayList.Add(Common.Utility.AppendLastUpdateInformation(insertSpecialInstructions, lastUpdateFunctionId));
			_sprocArrayList.Add(Common.VbecsStoredProcs.InsertSpecialInstructions.StoredProcName);
			_dtArrayList.Add(Common.Utility.AppendLastUpdateInformation(updateSpecialInstructions, lastUpdateFunctionId));
			_sprocArrayList.Add(Common.VbecsStoredProcs.UpdateSpecialInstructions.StoredProcName);
			_dtArrayList.Add(Common.Utility.AppendLastUpdateInformation(insertTransfusionRequirements, lastUpdateFunctionId));
			_sprocArrayList.Add(Common.VbecsStoredProcs.InsertTransfusionRequirements.StoredProcName);
			_dtArrayList.Add(Common.Utility.AppendLastUpdateInformation(updateTransfusionRequirements, lastUpdateFunctionId));
			_sprocArrayList.Add(Common.VbecsStoredProcs.UpdateTransfusionRequirements.StoredProcName);
	
			return DAL.Patient.SaveSIsAndTRs(_sprocArrayList, _dtArrayList);
		}

		

		
		

		

		#region static ABO/Rh retrieval

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/12/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5793"> 
		///		<ExpectedInput>Valid PatientGuid</ExpectedInput>
		///		<ExpectedOutput>ABORh object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5794"> 
		///		<ExpectedInput>Invalid PatientGuid</ExpectedInput>
		///		<ExpectedOutput>Exception</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns the patient ABO/Rh
		/// </summary>
		/// <param name="patientGuid"></param>
		/// <returns></returns>
		public static BOL.AboRh GetPatientAboRh(Guid patientGuid)
		{
			DataRow dtRow = DAL.Patient.GetPatientABORh(patientGuid);
			Common.ABO abo = Common.Utility.GetAboFromString(dtRow[Common.DatabaseConstants.ArtificialColumnNames.ABO].ToString());
			Common.RH rh = Common.Utility.GetRhFromString(dtRow[Common.DatabaseConstants.ArtificialColumnNames.Rh].ToString());
			BOL.AboRh aboRh = new AboRh(abo, rh);
			return aboRh;
		}

		/// <summary>
		/// GetJustification
		/// </summary>
		/// <param name="conversionABORh"></param>
		/// <param name="patientGuid"></param>
		/// <returns>ArrayList</returns>
		private static ArrayList GetJustification(BOL.AboRh conversionABORh, Guid patientGuid)
		{
			ArrayList aboHistory = new ArrayList();
			DataTable dt =  DAL.Patient.GetRecentPatientAboRHHistory(patientGuid);
			DataRow [] drABOs = dt.Select(VbecsTables.BloodTestType.BloodTestTypeId + " = " + (int) Common.TestType.ABORepeat + " OR BloodTestTypeId = " + (int) Common.TestType.ABOInterpTAS + " OR BloodTestTypeId = " + (int) Common.TestType.ABOInterp, "TestDate DESC");
			DataRow [] drRhs = dt.Select(VbecsTables.BloodTestType.BloodTestTypeId + " = " + (int) Common.TestType.RhRepeat + " OR BloodTestTypeId = " + (int) Common.TestType.RhInterpTAS  + " OR BloodTestTypeId = " + (int) Common.TestType.RhInterp, "TestDate DESC");

			// if 0 results, there is no inconsistency and therefore, no justification
			if (dt.Rows.Count == 0)
			{
				return aboHistory;
			}
				// When row count equals 2, compare specimen test with patient record
			else if(dt.Rows.Count==2 && drABOs.Length==1 && drRhs.Length==1 && conversionABORh != null)
			{
				Common.ABO testABO = Common.Utility.GetAboFromString( drABOs[0][VbecsTables.BloodUnitTest.TestResultId].ToString().Trim() );
				Common.RH testRh = Common.Utility.GetRhFromString( drRhs[0][VbecsTables.BloodUnitTest.TestResultId].ToString().Trim() );
				
				// If it's already been justified, return an empty ArrayList
				if ( (bool) drABOs[0][VbecsTables.SpecimenTest.AboRhChangeIndicator] == true) return aboHistory; 

				if ( (conversionABORh.Abo != Common.ABO.NA && 
					(conversionABORh.RH != Common.RH.Blank || conversionABORh.RH != Common.RH.NotProvided)) &&
					(conversionABORh.Abo != testABO || conversionABORh.RH != testRh))
				{
					aboHistory.Add(new SpecimenTest(drABOs[0]));
					aboHistory.Add(new SpecimenTest(drRhs[0]));
				}
			}
				// When row count equals 4, compare specimen tests 
			else if (dt.Rows.Count==4 && drABOs.Length==2 && drRhs.Length==2)
			{
				Common.ABO currentABO = Common.Utility.GetAboFromString( drABOs[0][VbecsTables.BloodUnitTest.TestResultId].ToString().Trim() );
				Common.RH currentRh = Common.Utility.GetRhFromString( drRhs[0][VbecsTables.BloodUnitTest.TestResultId].ToString().Trim() );
				
				// If it's already been justified, return an empty ArrayList
				if ( (bool) drABOs[0][VbecsTables.SpecimenTest.AboRhChangeIndicator] == true) return aboHistory; 

				if ( currentABO != Common.Utility.GetAboFromString( drABOs[1][VbecsTables.BloodUnitTest.TestResultId].ToString().Trim()) ||
					currentRh != Common.Utility.GetRhFromString( drRhs[1][VbecsTables.BloodUnitTest.TestResultId].ToString().Trim()) )
				{
					aboHistory.Add(new SpecimenTest(drABOs[0]));
					aboHistory.Add(new SpecimenTest(drRhs[0]));
					aboHistory.Add(new SpecimenTest(drABOs[1]));
					aboHistory.Add(new SpecimenTest(drRhs[1]));
				}
			}
			return aboHistory;
		}

		#endregion

		#endregion
	}
}




